#include <config.h>

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include "zvector.h"
#include "lapack.h"

static int totalzvectordata = 0;

int gettotalZVectordata()
{
	return totalzvectordata;
}



/* ZVector routines */

ZVector newZVector(int n)
{
	ZVector v;

	v = g_new(ZVectorType, n+1);
	v++;
	ZVectorSize(v) = n;

	totalzvectordata += n;

	return v;
}

void zeroZVector(ZVector v)
{
	int i, m;

	m = ZVectorSize(v);
	for(i = 0; i < m; i++) v[i].re = v[i].im = 0.0;
}

void fillZVector(ZVector v, double re, double im)
{
	int i, m;

	m = ZVectorSize(v);
	for(i = 0; i < m; i++) 
	{ 
		v[i].re = re; 
		v[i].im = im; 
	}
}

ZVector dupZVector(const ZVector v)
{
	int i, m;
	ZVector v2;

	m = ZVectorSize(v);
	v2 = newZVector(m);
	for(i = 0; i < m; i++) v2[i] = v[i];

	return v2;
}

void deleteZVector(ZVector v)
{
	totalzvectordata -= ZVectorSize(v);
	g_free(v-1);
}

int ZVectorisfinite(const ZVector v)
{
	int i, m;

	m = ZVectorSize(v);
	for(i = 0; i < m; i++) 
		if(!finite(v[i].re) || !finite(v[i].im)) return 0;

	return 1;
}

void subfromZVector(ZVector a, const ZVector b)
{
	int i, m, n;

	m = ZVectorSize(a);
	n = ZVectorSize(b);
	g_assert(m == n);

	for(i = 0; i < m; i++) 
	{ 
		a[i].re -= b[i].re; 
		a[i].im -= b[i].im; 
	}
}

void addtoZVector(ZVector a, const ZVector b)
{
	int i, m;

	g_assert(ZVectorSizeSame(a, b));
	
	m = ZVectorSize(a);

	for(i = 0; i < m; i++) 
	{
		a[i].re += b[i].re;
		a[i].im += b[i].im;
	}
}

void addscaledZVectortoZVector(ZVector a, const ZVector b, 
	double re_scale, double im_scale)
{
	int i, m;

	g_assert(ZVectorSizeSame(a, b));
	
	m = ZVectorSize(a);

	for(i = 0; i < m; i++) 
	{
		a[i].re += re_scale*b[i].re - im_scale*b[i].im;
		a[i].im += re_scale*a[i].im + im_scale*b[i].re;
	}
}

ZVector addZVectors(const ZVector a, const ZVector b)
{
	int i, m;
	ZVector Sum;

	g_assert(ZVectorSizeSame(a, b));
	
	m = ZVectorSize(a);

	Sum = newZVector(m);
	for(i = 0; i < m; i++) 
	{
		Sum[i].re = a[i].re + b[i].re;
		Sum[i].im = a[i].im + b[i].im;
	}
	
	return Sum;
}

ZVector subZVectors(const ZVector a, const ZVector b)
{
	int i, m;
	ZVector Dif;

	g_assert(ZVectorSizeSame(a, b));
	
	m = ZVectorSize(a);

	Dif = newZVector(m);
	for(i = 0; i < m; i++) 
	{
		Dif[i].re = a[i].re - b[i].re;
		Dif[i].im = a[i].im - b[i].im;
	}

	return Dif;
}

void copytoZVector(ZVector a, const ZVector b)
{
	int i, m;

	g_assert(a);
	g_assert(b);
	g_assert(ZVectorSizeSame(a, b));

	m = ZVectorSize(a);

	for(i = 0; i < m; i++) 
	{
		a[i].re = b[i].re;
		a[i].im = b[i].im;
	}
}

void scaleZVector(ZVector v, double re, double im)
{
	int i, m;
	double t;

	m = ZVectorSize(v);

	for(i = 0; i < m; i++) 
	{
		t = v[i].re;
		v[i].re = re*v[i].re - im*v[i].im;
		v[i].im = re*v[i].im + im*t;
	}
}

void biasZVector(ZVector v, double re, double im)
{
	int i, m;

	m = ZVectorSize(v);

	for(i = 0; i < m; i++) 
	{
		v[i].re += re;
		v[i].im += im;
	}
}

void printZVector(const ZVector v)
{
	int i, m;

	m = ZVectorSize(v);

	printf("ZVector(%d):\n", m);

	for(i = 0; i < m; i++)
	{
		printf("    %d %e %c %ei\n", i, v[i].re, 
			(v[i].im < 0 ? '-' : '+'), fabs(v[i].im));
	}
}

void saveZVectorasascii(const ZVector v, const char *filename)
{
	FILE *out;
	int i, m;

	g_assert(v);
	out = fopen(filename, "w");
	g_assert(out);
	m = ZVectorSize(v);

	for(i = 0; i < m; i++) fprintf(out, "%d %e %e\n", i, v[i].re, v[i].im);

	fclose(out);
}

void saveZVectorasformattedascii(const ZVector v, const char *filename,
	const char *format)
{
	FILE *out;
	int i, m;

	g_assert(v);
	out = fopen(filename, "w");
	g_assert(out);
	m = ZVectorSize(v);

	for(i = 0; i < m; i++) 
	{
		fprintf(out, "%d ", i);
		fprintf(out, format, v[i].re);
		fprintf(out, " ");
		fprintf(out, format, v[i].im);
		fprintf(out, "\n");
	}

	fclose(out);
}

ZVector ZVectorfromfile(const char *filename)
{
	FILE *in;
	char buf[1000];
	int i, n, a, bad = 0;
	double b, c;
	ZVector z;

	in = fopen(filename, "r");
	if(!in) return 0;

	for(n = 0; ; n++)
	{
		fgets(buf, 999, in);
		if(feof(in)) break;
		if(sscanf(buf, "%d%lf%lf", &a, &b, &c) != 3)
		{
			fclose(in);
			fprintf(stderr, "error opening %s as ZVector\n",
				filename);
			return 0;
		}
		if(a != n && bad == 0)
		{
			bad = 1;
			fprintf(stderr, "Warning : ZVectorfromfile -- column 0 "
				"does not equal element number\n");
		}
	}

	/* if we got here, the file must be good.  read it back in */
	rewind(in);
	
	z = newZVector(n);
	for(i = 0; i < n; i++)
	{
		fgets(buf, 999, in);
		sscanf(buf, "%d%lf%lf", &a, &b, &c);
		z[i].re = b;
		z[i].im = c;
	}
	
	fclose(in);

	return z;
}

void ZVectorsavebinary(const ZVector v, const char *filename)
{
	FILE *out;
	int m;

	g_assert(v);
	if(strcmp(filename, "-") == 0) out = stdout;
	else out = fopen(filename, "w");
	if(!out)
	{
		fprintf(stderr, "ZVectorsavebinary : "
			"Error opening %s for write\n", filename);
		return;
	}
	m = ZVectorSize(v)*2;	/* print 2 times number of complex numbers */
	fwrite(&m, sizeof(int), 1, out);
	m /= 2;
	fwrite(v, sizeof(ZVectorType), m, out);
	if(strcmp(filename, "-") != 0) fclose(out);
}

ZVector ZVectorloadbinary(const char *filename)
{
	FILE *in;
	int m;
	ZVector v;

	if(strcmp(filename, "-") == 0) in = stdin;
	else in = fopen(filename, "r");
	if(!in)
	{
		fprintf(stderr, "ZVectorloadbinary : "
			"Error opening %s for read\n", filename);
		return 0;
	}
	if(fread(&m, sizeof(int), 1, in) < 1)
	{
		fprintf(stderr, "ZVectorloadbinary : "
			"Error reading from %s\n", filename);
		return 0;
	}
	g_assert(m > 0);
	m /= 2;
	v = newZVector(m);
	g_assert(v);
	if(fread(v, sizeof(ZVectorType), m, in) < m)
	{
		fprintf(stderr, "WARNING: Vectorloadbinary : "
			"partial vector read from %s\n", filename);
	}
	if(strcmp(filename, "-") != 0) fclose(in);
	return v;
}

#if 0
int stringtoZVector(const char *str, ZVector v)
{
	int n=0, l, i=0;
	double d;
	gchar *s;

	s = g_strdup(str);

	for(i = 0; s[i]; i++) 
	{
		if(s[i] >= '0' && s[i] <= '9') continue;
		if(s[i] == '.' || s[i] == '+' || s[i] == '-') continue;
		if((s[i] == 'e' || s[i] == 'E') &&
		   (s[i+1] == '+' || s[i+1] == '-')) continue;
		s[i] = ' ';
	}
	i = 0;

	while(sscanf(s+n, "%lf%n", &d, &l) > 0)
	{
		if(v) if(i < VectorSize(v)) v[i] = d;
		i++;
		n+=l;
	}

	g_free(s);

	return i;
}
#endif

#if 0
ZVector newZVectorfromstring(const char *str)
{
	Vector v;
	int n;
	
	n = stringtoVector(str, 0);
	if(n == 0) return 0;
	
	v = newVector(n);
	stringtoVector(str, v);

	return v;
}
#endif

#if 0
char *ZVectortostring(const ZVector v)
{
	GString *str;
	gchar *out;
	char temp[100];
	int i, n;

	g_assert(v);
	
	n = VectorSize(v);
	sprintf(temp, "%f", v[0]);
	str = g_string_new(temp);

	if(n > 1) for(i = 1; i < n; i++)
	{
		sprintf(temp, ", %f", v[i]);
		str = g_string_append(str, temp);
	}

	out = str->str;

	return out;
}
#endif

ZVector subZVector(const ZVector v, int n1, int n2)
{
	ZVector u;
	int i, m;
	
	g_assert(v);
	m = ZVectorSize(v);
	
	if(n1 < 0 || n2 < n1 || n2 > m)
	{
		fprintf(stderr, "subZVector : range error\n");
		return 0;
	}
	
	u = newZVector(n2-n1+1);
	for(i = n1; i <= n2; i++) u[i-n1] = v[i];

	return u;
}

void conjZVector(ZVector v)
{
	int i, n;

	g_assert(v);

	n = ZVectorSize(v);

	for(i = 0; i < n; i++) v[i].im = -v[i].im;
}

double ZVectorsumsquares(ZVector z)
{
	int i, n; 
	double sum=0.0;

	n = ZVectorSize(z);
	for(i = 0; i < n; i++)
		sum += z[i].re*z[i].re + z[i].im*z[i].im;

	return sum;
}

double ZVectornormalize(ZVector z)
{
	int i, n; 
	double norm=0.0, max=0.0, a=0.0, b=0.0;

	n = ZVectorSize(z);
	for(i = 0; i < n; i++)
	{
		norm = z[i].re*z[i].re + z[i].im*z[i].im;
		if(norm > max) 
		{
			max = norm;
			a = z[i].re;
			b = z[i].im;
		}
	}

	scaleZVector(z, a/max, -b/max);

	return sqrt(max);
}

/* ZMatrix Routines */

/* newZMatrix returns matrix with dimensions [n][m] */

ZMatrix newpaddedZMatrix(int n, int m, int rowpad)
{
	ZMatrix M;
	ZMatrixTypePointer D;
	int j, w;

	w = m+rowpad;
	D = g_new(ZMatrixType, w*n+ZMATRIXVALUESPERBLOCK)+ZMATRIXVALUESPERBLOCK;
	M = g_new(ZMatrixTypePointer, ((n>1) ? (n+ZMATRIXSTUBINDICES) : 
					       (2+ZMATRIXSTUBINDICES)));
	M += ZMATRIXSTUBINDICES;
	ZMatrixStub(M)->datastart = D;
	ZMatrixStub(M)->n = n;
	ZMatrixStub(M)->m = m;
	ZMatrixStub(M)->rowpad = rowpad;
	ZMatrixrefcount(M) = 1;
	for(j = 0; j < n; j++) M[j] = D + (j*w);  /* Index array for [][] */
	if(n == 1) M[1] = D + w;

	return M;
}

void deleteZMatrix(ZMatrix M)
{
	ZMatrixStubType *ms;
	
	g_assert(M);
	ZMatrixrefcount(M)--;
	ms = ZMatrixStub(M);
	if(ZMatrixrefcount(M) <= 0) 
	{
		if(ms->datastart-ZMATRIXVALUESPERBLOCK == 0)
			fprintf(stderr, "ACK!!! ZMatrix data gone!\n");
		else
			g_free(ms->datastart-ZMATRIXVALUESPERBLOCK);
	}
	g_free(ms);
}

ZMatrix refsubZMatrix(const ZMatrix M, int n1, int m1, int n2, int m2)
{
	ZMatrix N;
	int j, m, n, w;

	g_assert(M);

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	if(n1 < 0) n1 = 0;
	if(n2 < 0) n2 = n-1;
	if(m1 < 0) m1 = 0;
	if(m2 < 0) m2 = m-1;

	if(n2 < n1 || m2 < m1 || m2 >= m || n2 >= n)
	{
		fprintf(stderr, "refsubZMatrix: limit error\n");
		return 0;
	}

	w = ZMatrixstride(M);
	N = g_new(ZMatrixTypePointer, ((n>1) ? (n+ZMATRIXSTUBINDICES) : 
					       (2+ZMATRIXSTUBINDICES)));
	N += ZMATRIXSTUBINDICES;
	ZMatrixStub(N)->datastart = ZMatrixStub(M)->datastart;
	ZMatrixStub(N)->n = n2-n1+1;
	ZMatrixStub(N)->m = m2-m1+1;
	ZMatrixStub(N)->rowpad = w-(m2-m1+1);
	ZMatrixrefcount(M)++;
	for(j = 0; j <= n2-n1; j++) N[j] = M[n1] + m1 + (j*w);  
	if(n2 == n1) N[1] = N[0]+w;

	return N;
}

void ZMatrixsavebinary(const ZMatrix M, const char *filename)
{
	FILE *out;
	int m, n, p;

	g_assert(M);
	if(strcmp(filename, "-") == 0) out = stdout;
	else out = fopen(filename, "w");
	if(!out)
	{
		fprintf(stderr, "ZMatrixsavebinary : "
			"Error opening %s for write\n", filename);
		return;
	}
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M) * 2;	/* so format is compatible to Matrix */
	p = ZMatrixrowpad(M) * 2;
	fwrite(&n, sizeof(int), 1, out);
	fwrite(&m, sizeof(int), 1, out);
	fwrite(&p, sizeof(int), 1, out);
	m /= 2;
	p /= 2;
	fwrite(M[0], sizeof(ZMatrixType), n*(m+p), out);
	if(strcmp(filename, "-") != 0) fclose(out);
}

ZMatrix ZMatrixloadbinary(const char *filename)
{
	FILE *in;
	int m, n, p;
	ZMatrix M;

	if(strcmp(filename, "-") == 0) in = stdin;
	else in = fopen(filename, "r");
	if(!in)
	{
		fprintf(stderr, "ZMatrixloadbinary : "
			"Error opening %s for read\n", filename);
		return 0;
	}
	if(!fread(&n, sizeof(int), 1, in) ||
	   !fread(&m, sizeof(int), 1, in) ||
	   !fread(&p, sizeof(int), 1, in))
	{
		fprintf(stderr, "Matrixloadbinary : Error reading from %s .\n", filename);
		return 0;
	}
	m /= 2;
	p /= 2;
	M = newpaddedZMatrix(n, m, p);
	g_assert(M);
	if(fread(M[0], sizeof(ZMatrixType), n*(m+p), in) < n*(m+p))
		fprintf(stderr, "WARNING: ZMatrixloadbinary : "
			"partial matrix loaded from %s\n", filename);

	return M;
}

void zeroZMatrix(ZMatrix M)
{
	int i, j, m, n;
	ZMatrixTypePointer Mrow;

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	for(j = 0; j < n; j++) 
	{
		Mrow = M[j];
		for(i = 0; i < m; i++) Mrow[i].re = Mrow[i].im = 0.0;
	}
}

ZMatrix dupZMatrix(const ZMatrix M)
{
	int i, j, m, n;
	ZMatrix M2;

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	M2 = newpaddedZMatrix(n, m, ZMatrixrowpad(M));
	for(i = 0; i < m; i++) for(j = 0; j < n; j++) M2[j][i] = M[j][i];

	return M2;
}

void printZMatrix(const ZMatrix M)
{
	int i, j, n, m;
	
	if(!M)
	{
		printf("printMatrix: NULL\n");
		return;
	}
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	
	printf("ZMatrix (%d by %d):\n", n, m);
	
	for(j = 0; j < n; j++)
	{
		for(i = 0; i < m; i++) printf("(%+5e, %+5e) ", 
			M[j][i].re, M[j][i].im);
		printf("\n");
	}
}

ZMatrix repadZMatrix(const ZMatrix M, int pad)
{
	int i, j, m, n;
	ZMatrix M2;

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	M2 = newpaddedZMatrix(n, m, pad);
	for(i = 0; i < m; i++) for(j = 0; j < n; j++) M2[j][i] = M[j][i];

	return M2;
}

void scaleZMatrix(ZMatrix M, double re, double im)
{
	int i, j, m, n;
	double t;
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	
	for(j = 0; j < n; j++) for(i = 0; i < m; i++) 
	{
		t = M[j][i].re;
		M[j][i].re = re*M[j][i].re - im*M[j][i].im;
		M[j][i].im = re*M[j][i].im + im*t;
	}
}

void addscaledZMatrixtoZMatrix(ZMatrix a, const ZMatrix b, double re, double im)
{
	int i, j, m, n;

	g_assert(ZMatrixSizeSame(a, b));

	n = ZMatrixSize1(a);
	m = ZMatrixSize2(a);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
	{
		a[j][i].re += re*b[j][i].re - im*b[j][i].im;
		a[j][i].im += re*b[j][i].im - im*b[j][i].re;
	}
}

void biasZMatrix(ZMatrix M, double re, double im)
{
	int i, j, m, n;
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	
	for(j = 0; j < n; j++) for(i = 0; i < m; i++) 
	{
		M[j][i].re += re;
		M[j][i].im += im;
	}
}

void addtoZMatrix(ZMatrix M, const ZMatrix N)
{
	int i, j, m, n;
	
	g_assert(ZMatrixSizeSame(M, N));
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
	{
		M[j][i].re += N[j][i].re;
		M[j][i].im += N[j][i].im;
	}
}

/* V[i] <- sum_j M[i][j] V[j] */
void multiplyZMatrixtoZVector(const ZMatrix M, ZVector V)
{
        int m, n;
        int i, j;
        ZVector v;

        g_assert(M);
        g_assert(V);

        m = ZMatrixSize1(M);
        n = ZMatrixSize2(M);
        g_assert(ZVectorSize(V) == n);

        v = newZVector(m);
        zeroZVector(v);

        for(i = 0; i < m; i++) for(j = 0; j < n; j++) 
	{
		v[i].re += M[i][j].re*V[j].re - M[i][j].im*V[j].im;
		v[i].im += M[i][j].re*V[j].im + M[i][j].im*V[j].re;
	}

        copytoZVector(V, v);
        deleteZVector(v);
}

/* call with a guess for v */
double ZMatrixgreatesteigenvector(const ZMatrix M, ZVector v, int niter)
{
	int i;
	double s=0;

	g_assert(M);
	g_assert(v);

	for(i = 0; i < niter; i++)
	{
		multiplyZMatrixtoZVector(M, v);
		s = ZVectornormalize(v);
	}

	return s;
}

void subfromZMatrix(ZMatrix M, const ZMatrix N)
{
	int i, j, m, n;
	
	g_assert(ZMatrixSizeSame(M, N));
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
	{
		M[j][i].re -= N[j][i].re;
		M[j][i].im -= N[j][i].im;
	}
}

ZMatrix addZMatrices(const ZMatrix M, const ZMatrix N)
{
	int i, j, m, n;
	ZMatrix Sum;
	
	g_assert(ZMatrixSizeSame(M, N));
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	Sum = newZMatrix(n, m);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
	{
		Sum[j][i].re = M[j][i].re + N[j][i].re;
		Sum[j][i].im = M[j][i].im + N[j][i].im;
	}

	return Sum;
}

void copytoZMatrix(ZMatrix M, const ZMatrix N)
{
	int i, j, m, n;
	
	g_assert(ZMatrixSizeSame(M, N));
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
	{
		M[j][i].re = N[j][i].re;
		M[j][i].im = N[j][i].im;
	}
}

ZMatrix dupsubZMatrix(const ZMatrix M, int n1, int m1, int n2, int m2)
{
	ZMatrix N;
	int i, j, m, n;

	g_assert(M);

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	if(n1 < 0) n1 = 0;
	if(n2 < 0) n2 = n-1;
	if(m1 < 0) m1 = 0;
	if(m2 < 0) m2 = m-1;

	if(n2 < n1 || m2 < m1 || m2 >= m || n2 >= n)
	{
		fprintf(stderr, "dupsubMatrix: limit error\n");
		return 0;
	}

	N = newZMatrix(n2-n1+1, m2-m1+1);
	
	for(j = n1; j <= n2; j++) for(i = m1; i <= m2; i++)
		N[j-n1][i-m1] = M[j][i];

	return N;
}

/* Copy region (m1, n1)-(m2, n2) from M to region (m3, n3) in N */
void copysubZMatrix(ZMatrix N, const ZMatrix M, int n1, int m1, int n2, int m2,
	int n3, int m3)
{
	int i, j, m, n;

	g_assert(M);
	g_assert(N);

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	if(n1 < 0) n1 = 0;
	if(n2 < 0) n2 = n-1;
	if(m1 < 0) m1 = 0;
	if(m2 < 0) m2 = m-1;

	if(n2 < n1 || m2 < m1 || m2 >= m || n2 >= n ||
		n3 >= ZMatrixSize1(N) || m3 >= ZMatrixSize2(N))
	{
		fprintf(stderr, "copysubZMatrix: limit error\n");
		return;
	}

	if(n3 < 0) n3 = ZMatrixSize1(N)-(n2-n1);
	if(m3 < 0) m3 = ZMatrixSize2(N)-(m2-m1);

	if(n3 < 0)
	{
		n1 -= n3;
		n3 = 0;
	}
	if(m3 < 0)
	{
		m1 -= m3;
		m3 = 0;
	}
	
	if(n3+n2-n1 >= ZMatrixSize1(N))
		n2 = n1-n3+ZMatrixSize1(N)-1;
	if(m3+m2-m1 >= ZMatrixSize2(N))
		m2 = m1-m3+ZMatrixSize2(N)-1;

	for(j = n1; j <= n2; j++) for(i = m1; i <= m2; i++)
		N[j+n1-n3][i+m1-m3] = M[j][i];
}

ZVector ZMatrixrow(const ZMatrix M, int row)
{
	ZVector V;
	int m, n, i;
	ZMatrixTypePointer Mrow;

	g_assert(M);
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	if(row < 0) row = n-row;
	g_assert((row < n) && (row >= 0));
	Mrow = M[row];

	V = newZVector(m);
	
	for(i = 0; i < m; i++) V[i] = Mrow[i];

	return V;
}

ZVector ZMatrixcolumn(const ZMatrix M, int column)
{
	ZVector V;
	int m, n, j;
	
	g_assert(M);
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);
	if(column < 0) column = m-column;
	g_assert((column < m) && (column >= 0));

	V = newZVector(n);

	for(j = 0; j < n; j++) V[j] = M[j][column];

	return V;
}

/* C[i][k] = sum_j A[i][j] B[j][k] */
ZMatrix ZMatrixmultiply(const ZMatrix A, const ZMatrix B)
{
	int i, j, k, l, m, n;
	ZMatrix C;
	
	g_assert(ZMatrixSize2(A) == ZMatrixSize1(B));

	l = ZMatrixSize1(A);
	m = ZMatrixSize2(A);
	n = ZMatrixSize2(B);

	C = newZMatrix(l, n);
	zeroZMatrix(C);
	
	for(k = 0; k < n; k++) for(j = 0; j < m; j++) for(i = 0; i < l; i++)
	{	
		C[i][k].re += A[i][j].re*B[j][k].re - A[i][j].im*B[j][k].im;
		C[i][k].im += A[i][j].re*B[j][k].im + A[i][j].im*B[j][k].re;
	}

	return C;
}

/* C[i][k] = sum_j A[i][j] B[j][k] */
void copyZMatrixmultiply(ZMatrix C, const ZMatrix A, const ZMatrix B)
{
	int i, j, k, l, m, n;
	
	g_assert(ZMatrixSize2(A) == ZMatrixSize1(B));
	g_assert(ZMatrixSize1(C) == ZMatrixSize1(A));
	g_assert(ZMatrixSize2(C) == ZMatrixSize2(B));

	l = ZMatrixSize1(A);
	m = ZMatrixSize2(A);
	n = ZMatrixSize2(B);

	zeroZMatrix(C);
	
	for(k = 0; k < n; k++) for(j = 0; j < m; j++) for(i = 0; i < l; i++)
	{
		C[i][k].re += A[i][j].re*B[j][k].re - A[i][j].im*B[j][k].im;
		C[i][k].im += A[i][j].re*B[j][k].im + A[i][j].im*B[j][k].re;
	}
}

ZVector ZMatrixZVectormultiply(const ZMatrix M, const ZVector V)
{
	int m, n;
	int i, j;
	ZVector v;
	
	g_assert(M);
	g_assert(V);
	
	m = ZMatrixSize1(M);
	n = ZMatrixSize2(M);
	g_assert(ZVectorSize(V) == n);

	v = newZVector(m);
	zeroZVector(v);

	for(i = 0; i < m; i++) for(j = 0; j < n; j++) 
	{
		v[i].re += M[i][j].re*V[j].re - M[i][j].im*V[j].im;
		v[i].im += M[i][j].re*V[j].im + M[i][j].im*V[j].re;
	}

	return v;
}

ZMatrix rollZMatrix(const ZMatrix M, int delta_n, int delta_m)
{
	ZMatrix N;
	int i, j, m, n;

	g_assert(M);

        n = ZMatrixSize1(M);
        m = ZMatrixSize2(M);

        N = newZMatrix(n, m);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
                N[j][i] = M[(j + delta_n) % n][(i + delta_m) % m];

	return N;
}

void rollZMatrixinplace(ZMatrix M, int delta_n, int delta_m)
{
	ZVector v;
	int rowcount;
	int startrow = 0, row;

        int i, m, n;                

        g_assert(M);

        n = ZMatrixSize1(M);
        m = ZMatrixSize2(M);

	delta_n = (n+delta_n) % n;
	delta_m = (m+delta_m) % m;

	v = ZMatrixrow(M, 0);
	row = (n-delta_n) % n;

	for(rowcount = n-1; rowcount >= 0; rowcount--)
	{
		if(row == startrow)
		{
			for(i = 0; i < m; i++) 
				M[(row + delta_n) % n][(i+delta_m) % m] = v[i];
			startrow = row = (row + 1) % n;
			if(rowcount) 
				for(i = 0; i < m; i++) v[i] = M[row][i];
		}
		else for(i = 0; i < m; i++) 
			M[(row + delta_n) % n][(i + delta_m) % m] = M[row][i];

		row = (row - delta_n + n) % n;
	}

	deleteZVector(v);
}

ZMatrix transposeZMatrix(const ZMatrix M)
{
	ZMatrix N;
	int i, j, m, n;
	
	g_assert(M);

	n = ZMatrixSize1(M);
        m = ZMatrixSize2(M);

	N = newZMatrix(m, n);

	for(j = 0; j < n; j++) for(i = 0; i < m; i++)
		N[i][j] = M[j][i];

	return N;
}

void transposeZMatrixinplace(ZMatrix M)
{
	int i, j, n;
	ZMatrixType t;

	g_assert(M);

	n = ZMatrixSize1(M);

	if(n != ZMatrixSize2(M))
	{
		fprintf(stderr, "transposeZMatrixinplace : not square\n");
		return;
	}

	for(j = 1; j < n; j++) for(i = 0; i < j; i++)
	{
		t = M[i][j];
		M[i][j] = M[j][i];
		M[j][i] = t;
	}
}

ZMatrix conjZMatrix(const ZMatrix M)
{
	ZMatrix N;
	int i, j, n, m, p;

	g_assert(M);

	n = ZMatrixSize1(M);
        m = ZMatrixSize2(M);
	p = ZMatrixrowpad(M);

	N = newpaddedZMatrix(n, m, p);

	for(j = 0; j < n; j++) for(i = 0; i < m+p; i++)
	{
		N[j][i].re =  M[j][i].re;
		N[j][i].im = -M[j][i].im;
	}
	
	return N;
}

void conjZMatrixinplace(const ZMatrix M)
{
	int i, j, n, m;

	g_assert(M);

	n = ZMatrixSize1(M);
        m = ZMatrixSize2(M) + ZMatrixrowpad(M);

	for(j = 1; j < n; j++) for(i = 0; i < m; i++)
		M[j][i].im = -M[j][i].im;
}

int ZMatrixisHermitian(const ZMatrix M)
{
	int i, j, n;

	g_assert(M);

	n = ZMatrixSize1(M);

	g_assert(ZMatrixSize2(M) == n);

	for(i = 0; i < n; i++) 
		if(M[i][i].im != 0.0) return 0;
	for(i = 1; i < n; i++) for(j = 0; j < i; j++)
	{
		if(M[i][j].re != M[j][i].re) return 0;
		if(M[i][j].im != -M[j][i].im) return 0;
	}

	return 1;
}

int Zlinearsolve(ZMatrix a, ZVector b)
{
	ZVector c;

	ZMatrixLUinvert(a);
	c = ZMatrixZVectormultiply(a, b);
	copytoZVector(b, c);
	deleteZVector(c);

	return 1;
}
