#include <config.h>

#include <stdarg.h>
#include <glib.h>
#include <stdio.h>
#include <math.h>
#include "zvecarray.h"

ZVecArray newZVecArray(int n)
{
	ZVecArray a;
	int i;

	a = g_new(ZVectorTypePointer, n+1);
	a++;
	ZVecArraySize(a) = n;

	for(i = 0; i < n; i++) a[i] = 0;

	return a;
}

ZVecArray newpopulatedZVecArray(int n, int m)
{
	ZVecArray a;
	int i;

	a = g_new(ZVectorTypePointer, n+1);
	a++;
	ZVecArraySize(a) = n;

	for(i = 0; i < n; i++) a[i] = newZVector(m);

	return a;
}

void zeroZVecArray(ZVecArray a)
{
	int i, n;

	g_assert(a);

	n = ZVecArraySize(a);
	for(i = 0; i < n; i++) if(a[i]) zeroZVector(a[i]);
}

void deleteZVecArray(ZVecArray a)
{
	g_assert(a);

	ZVecArraySize(a) = 0;

	g_free(a-1);
}

void deleteZVecArrayandZVectors(ZVecArray a)
{
	int i, n;
	
	g_assert(a);

	n = ZVecArraySize(a);
	for(i = 0; i < n; i++) if(a[i]) deleteZVector(a[i]);
	ZVecArraySize(a) = 0;

	g_free(a-1);
}

ZVecArray dupZVecArray(const ZVecArray a)
{
	ZVecArray b;
	int i, n;

	g_assert(a);

	n = ZVecArraySize(a);
	b = newZVecArray(n);
	for(i = 0; i < n; i++) b[i] = a[i];

	return b;
}

ZVecArray dupZVecArrayandZVectors(const ZVecArray a)
{
	ZVecArray b;
	int i, n;

	g_assert(a);

	n = ZVecArraySize(a);
	b = newZVecArray(n);
	for(i = 0; i < n; i++) if(a[i]) b[i] = dupZVector(a[i]);

	return b;
}

void copytoZVecArray(ZVecArray dest, const ZVecArray src)
{
	int i, n;

	g_assert(dest);
	g_assert(src);

	n = ZVecArraySize(src);
	
	g_assert(n == ZVecArraySize(dest));
	
	for(i = 0; i < n; i++) copytoZVector(dest[i], src[i]);
}

ZVecArray ZVecArraysubset(const ZVecArray a, int col, int ncol)
{
	ZVecArray b;
	int i;

	g_assert(a);
	g_assert(col > 0);
	g_assert(ncol > 0);
	g_assert(col+ncol <= ZVecArraySize(a));

	b = newZVecArray(ncol);

	for(i = 0; i < ncol; i++) b[i] = a[i+col];

	return b;
}

ZVecArray dupZVecArraysubset(const ZVecArray a, int col, int ncol)
{
	ZVecArray b;
	int i;

	g_assert(a);
	g_assert(col >= 0);
	g_assert(ncol > 0);
	g_assert(col+ncol <= ZVecArraySize(a));

	b = newZVecArray(ncol);

	for(i = 0; i < ncol; i++) if(a[i]) b[i] = dupZVector(a[i+col]);

	return b;
}

ZVecArray dupZVecArrayrowsubset(const ZVecArray a, int row, int nrow)
{
	ZVecArray b;
	int N, i, j;

	g_assert(a);
	N = ZVecArraySize(a);
	g_assert(row >= 0);
	g_assert(nrow > 0);
	g_assert(row+nrow <= ZVecArrayZVectorSize(a));

	b = newpopulatedZVecArray(N, nrow);
	for(j = 0; j < nrow; j++) for(i = 0; i < N; i++)
		b[i][j] = a[i][j+row];

	return b;
}

int ZVecArrayZVectorSize(const ZVecArray a)
{
	int i, n, s=-1;

	g_assert(a);

	n = ZVecArraySize(a);
	for(i = 0; i < n; i++) if(a[i])
	{
		if(s < 0) s = ZVectorSize(a[i]);
		else if(s != ZVectorSize(a[i])) return -1;
	}

	return s;
}

int ZVecArrayZVectorcolumns(const ZVecArray a)
{
	int i, n, c=0;

	g_assert(a);

	n = ZVecArraySize(a);
	for(i = 0; i < n; i++) if(a[i]) c++;
	
	return c;
}

ZVecArray ZVecArraydeletecolumn(ZVecArray a, int col)
{
	ZVecArray b;
	int i, n;

	g_assert(a);
	
	n = ZVecArraySize(a);
	g_assert(col >= 0 && col < n);
	
	if(n <= 1)
	{
		deleteZVecArray(a);
		return 0;
	}

	b = newZVecArray(n-1);
	
	if(col) for(i = 0; i < col; i++) b[i] = a[i];
	if(col < n-1) for(i = col+1; i < n; i++) b[i-1] = a[i];

	deleteZVecArray(a);

	return b;
}

ZVecArray ZVecArrayinsertcolumn(ZVecArray a, int col)
{
	ZVecArray b;
	int i, n;
	
	g_assert(col >= 0);
	
	if(a == 0) return newZVecArray(1);

	n = ZVecArraySize(a);
	g_assert(col <= n);

	b = newZVecArray(n+1);
	
	if(col > 0) for(i = 0; i < col; i++) b[i] = a[i];
	if(col < n) for(i = col; i < n; i++) b[i+1] = a[i];

	deleteZVecArray(a);

	return b;
}

ZVecArray ZVecArrayappendcolumn(ZVecArray a)
{
	ZVecArray b;
	int i, n;

	if(!a) return newZVecArray(1);

	n = ZVecArraySize(a);
	b = newZVecArray(n+1);
	for(i = 0; i < n; i++) b[i] = a[i];

	deleteZVecArray(a);

	return b;
}

ZVecArray ZVecArrayinsertZVector(ZVecArray a, ZVector v, int col)
{
	ZVecArray b;

	b = ZVecArrayinsertcolumn(a, col);
	b[col] = v;

	return b;
}

ZVecArray ZVecArrayappendZVector(ZVecArray a, ZVector v)
{
	ZVecArray b;
	int i, n;

	if(!a)
	{
		b = newZVecArray(1);
		b[0] = v;
		return b;
	}

	n = ZVecArraySize(a);
	b = newZVecArray(n+1);
	for(i = 0; i < n; i++) b[i] = a[i];

	deleteZVecArray(a);

	b[n] = v;

	return b;
}

ZVecArray ZVecArrayappendZVectors(ZVecArray a, ...) 
{
	va_list ap;
	ZVector v;

	va_start(ap, a);
	while((v = va_arg(ap, ZVector)))
		a = ZVecArrayappendZVector(a, v);
	va_end(ap);

	return a;
}

/*
VecArray newVecArrayfromVectors(...)
{
	va_list ap;
	VecArray a = 0;
	Vector v;

	va_start(ap, a);
	while((v = va_arg(ap, Vector)))
		a = VecArrayappendVector(a, v);
	va_end(ap);

	return a;
}
*/

void ZVecArraysetcolumn(ZVecArray a, ZVector v, int col)
{
	g_assert(a);
	g_assert(col >= 0 && col < ZVecArraySize(a));
	
	a[col] = v;
}

ZMatrix newZMatrixfromZVecArray(const ZVecArray a)
{
	ZMatrix M;
	int m, n, i, j;

	g_assert(a);

	m = ZVecArrayZVectorSize(a);
	if(m <= 0) return 0;
	n = ZVecArraySize(a);
	
	M = newZMatrix(n, m);
	for(j = 0; j < n; j++) for(i = 0; i < m; i++) M[j][i] = a[j][i];

	return M;
}

ZVecArray newZVecArrayfromZMatrix(const ZMatrix M)
{
	int i, j, m, n;
	ZVecArray a;

	g_assert(M);
	
	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	a = newZVecArray(n);
	for(j = 0; j < n; j++) 
	{
		a[j] = newZVector(m);
		for(i = 0; i < m; i++) a[j][i] = M[j][i];
	}

	return a;
}

ZVecArray ZVecArrayconcat(ZVecArray a, ZVecArray b)
{
	ZVecArray c;
	int i, na, nb;

	g_assert(a);
	g_assert(b);
	
	na = ZVecArraySize(a);
	nb = ZVecArraySize(b);

	c = newZVecArray(na+nb);
	for(i = 0; i < na; i++) c[i] = a[i];
	for(i = 0; i < nb; i++) c[i+na] = b[i];

	deleteZVecArray(a);
	deleteZVecArray(b);

	return c;
}

ZVecArray ZVecArrayappendZVecArray(ZVecArray a, const ZVecArray b)
{
	ZVecArray c;
	int i, na, nb;

	g_assert(a);
	g_assert(b);
	
	na = ZVecArraySize(a);
	nb = ZVecArraySize(b);

	c = newZVecArray(na+nb);
	for(i = 0; i < na; i++) c[i] = a[i];
	for(i = 0; i < nb; i++) c[i+na] = b[i];

	deleteZVecArray(a);

	return c;
}

ZVecArray ZVecArrayappendZVecArrays(ZVecArray a, ...) 
{
	va_list ap;
	ZVecArray b;

	va_start(ap, a);
	while((b = va_arg(ap, ZVecArray)))
		a = ZVecArrayappendZVecArray(a, b);
	va_end(ap);

	return a;
}

ZVecArray reduceZVecArraybymask(ZVecArray a, const intVector mask)
{
	ZVecArray b;
	int i, m, j, n, c=0;
	
	g_assert(a);
	g_assert(mask);
	
	m = ZVecArrayZVectorSize(a);
	g_assert(m == ZVectorSize(mask));

	n = ZVecArraySize(a);
	
	for(i = 0; i < m; i++) if(mask[i] > 0) c++;
	
	b = newpopulatedZVecArray(n, c);

	c = 0;

	for(i = 0; i < m; i++) if(mask[i])
	{
		for(j = 0; j < n; j++) b[j][c] = a[j][i];
		c++;
	}

	deleteZVecArrayandZVectors(a);

	return b;
}

#if 0
ZVecArray ZVecArrayfromfile(const char *filename, int mincols)
{
	ZVecArray a;
	ZVector v;
	FILE *in;
	int nc = 0, nr = 0, n, i, row;
	char line[1024];
	
	in = fopen(filename, "r");
	if(!in) return 0;

	for(;;)
	{
		fgets(line, 1023, in);
		if(feof(in)) break;
		/* look for "standard" comment characters */
		if(line[0] == '#' || line[0] > 57 || line[0] == '*' 
				  || line[0] == '!' || line[0] == ';') continue;
		n = stringtoZVector(line, 0);
		if(n >= mincols)
		{
			if(nc == 0) nc = n;
			if(n >= nc) nr++;
		}
	}
	fclose(in);

	a = newpopulatedZVecArray(nc, nr);
	v = newZVector(nc);

	in = fopen(filename, "r");
	for(row = 0; row < nr;)
	{
		fgets(line, 1023, in);
		if(feof(in)) break;
		if(line[0] == '#' || line[0] > 57 || line[0] == '*') continue;
		n = stringtoZVector(line, v);
		if(n >= nc)
		{
			if(n >= nc)
			{
				for(i = 0; i < nc; i++) a[i][row] = v[i];
				row++;
			}
		}
	}
	fclose(in);

	deleteZVector(v);

	return a;
}
#endif

int ZVecArraytofile(const ZVecArray a, const char *filename, 
	const char *format)
{
	int m, n, i, j, needspace = 0;
	const char *f;
	FILE *out;

	g_assert(a);

	if(format == 0) f = "%10.5f";
	else f = format;

	m = ZVecArrayZVectorSize(a);
	if(m <= 0) 
	{
		fprintf(stderr, "ZVecArraycolumnstofile : "
			"ZVecArray is too complicated\n");
		return -1;
	}
	n = ZVecArraySize(a);

	out = fopen(filename, "w");
	if(!out)
	{
		fprintf(stderr, "ZVecArraycolumnstofile : "
			"Cannot open file %s for write\n", filename);
		return -1;
	}

	for(i = 0; i < m; i++) 
	{
		needspace = 0;
		for(j = 0; j < n; j++) if(a[j]) 
		{
			if(needspace++) fprintf(out, " ");
			fprintf(out, f, a[j][i].re);
			fprintf(out, " ");
			fprintf(out, f, a[j][i].im);
		}
		fprintf(out, "\n");
	}

	fclose(out);

	return needspace;
}

void ZVecArrayresize(const ZVecArray a, int newsize)
{
	int i, j, m, nr, nc;
	ZVector V;
	
	g_assert(a);
	
	nc = ZVecArraySize(a);
	nr = ZVecArrayZVectorSize(a);

	g_assert(nr);
	g_assert(nc);

	if(newsize < nr) m = newsize;
	else m = nr;

	for(i = 0; i < nc; i++)
	{
		V = newZVector(newsize);
		for(j = 0; j < m; j++) V[j] = a[i][j];
		if(newsize > nr)
			for(j = nr; j < newsize; j++) 
				a[i][j].re = a[i][j].im = 0.0;
		deleteZVector(a[i]);
		a[i] = V;
	}
}

/* Arithmetic operations */

void scaleZVecArray(ZVecArray a, double re, double im)
{
	int N, j;

	g_assert(a);

	N = ZVecArraySize(a);
	for(j = 0; j < N; j++) if(a[j]) scaleZVector(a[j], re, im);
}

void addtoZVecArray(ZVecArray a, const ZVecArray b)
{
	int i, M;
	
	g_assert(a);
	g_assert(b);
	
	M = ZVecArraySize(a);
	g_assert(M == ZVecArraySize(a));

	for(i = 0; i < M; i++) addtoZVector(a[i], b[i]);
}

ZVector ZVecArrayZVectoraverage(const ZVecArray a)
{
	int N, M, j;
	ZVector v;

	g_assert(a);

	N = ZVecArrayZVectorSize(a);
	if(N <= 0) return 0;

	M = ZVecArraySize(a);

	v = newZVector(N);
	zeroZVector(v);
	for(j = 0; j < M; j++) addtoZVector(v, a[j]);
	scaleZVector(v, 1.0/M, 0.0);

	return v;
}


