#include <stdio.h>
#include <glib.h>
#include <string.h>
#include "uvfits.h"
#include "keyvalue.h"
#include "byteorder.h"

int UVFitsparseFQtable(UVFits UV)
{
	FitsTable T;
	int n, ret, axislen;
	double reffreq;
	
	g_assert(UV);

	T = FitsgetFitsTablebyname((Fits)UV, "AIPS FQ", 1);
	if(!T)
	{
		fprintf(stderr, "AIPS FQ not found\n");
		return 0;
	}

	if(UV->fqtab.freq) g_free(UV->fqtab.freq);
	if(UV->fqtab.bw) g_free(UV->fqtab.cw);
	if(UV->fqtab.cw) g_free(UV->fqtab.bw);
	if(UV->fqtab.sideband) g_free(UV->fqtab.sideband);
	
	UV->fqtab.nfreq = getKeyValueint(T->kv, "NO_IF");
	g_assert(UV->fqtab.nfreq != KV_INTERR);

	UV->fqtab.freq = FitsTablegetdatadouble(T, "IF FREQ",         0, &n);
	g_assert(n >= UV->fqtab.nfreq);
	UV->fqtab.bw   = FitsTablegetdatafloat (T, "TOTAL BANDWIDTH", 0, &n);
	g_assert(n >= UV->fqtab.nfreq);
	UV->fqtab.cw   = FitsTablegetdatafloat (T, "CH WIDTH",        0, &n);
	g_assert(n >= UV->fqtab.nfreq);
	UV->fqtab.sideband=FitsTablegetdataint (T, "SIDEBAND",        0, &n);
	g_assert(n >= UV->fqtab.nfreq);
	
	ret = FitsTablegetaxisinfo(((Fits)UV)->tables[0], 
		"FREQ", &axislen, &reffreq, 0, 0, 0);
	g_assert(ret == 1);
	g_assert(axislen == 1);

	fprintf(stderr, "reffreq = %f\n", reffreq);

	for(n = 0; n < UV->fqtab.nfreq; n++) 
	{
		UV->fqtab.freq[n] += reffreq;
		if(UV->fqtab.sideband[n] == 1)
			UV->fqtab.freq[n] +=  0.5*UV->fqtab.bw[n];
		else
			UV->fqtab.freq[n] -=  0.5*UV->fqtab.bw[n];
	}

	for(n = 0; n < UV->fqtab.nfreq; n++)
		printf("%f %f %f %d\n",	UV->fqtab.freq[n],
					UV->fqtab.bw[n],
					UV->fqtab.cw[n],
					UV->fqtab.sideband[n]);

	UVFitsgettimes(UV);

	return 1;
}

int UVFitsparseANtable(UVFits UV)
{
	FitsTable T;
	int a, n;
	
	g_assert(UV);

	T = FitsgetFitsTablebyname((Fits)UV, "AIPS AN", 1);
	if(!T)
	{
		fprintf(stderr, "AIPS AN not found\n");
		return 0;
	}

	if(UV->antab.name && UV->antab.nant > 0)
	{
		for(a = 0; a < UV->antab.nant; a++)
		if(UV->antab.name[a]) g_free(UV->antab.name[a]);
		g_free(UV->antab.name);
	}

	UV->antab.nant = getKeyValueint(T->kv, "NAXIS2");
	UV->antab.name = g_new(char *, UV->antab.nant);

	for(a = 0; a < UV->antab.nant; a++)
	{
		UV->antab.name[a] = FitsTablegetdatachar(T, "ANNAME", a, &n);
		g_assert(n > 0);
	}

	return 1;
}

int UVFitsgettimes(UVFits UV)
{
	int i;
	FitsTable T;
	float *d, t[2];
	int order[][2] = {{2, sizeof(float)}, {0, 0}};
	
	g_assert(UV);

	T = ((Fits)UV)->tables[0];

	if(UV->t) deleteVector(UV->t);

	UV->t = newVector(T->gcount);

	for(i = 0; i < T->pcount; i++) 
	{
		if(T->pdata[i].ptype == 0) continue;
		if(strncmp("DATE", T->pdata[i].ptype, 4) == 0) break;
	}

	if(i == T->pcount) return 0;

	d = (float *)T->data + i;

	for(i = 0; i < T->gcount; i++)
	{
		t[0] = d[i*UV->reclen];
		t[1] = d[i*UV->reclen+1];
		if(FitsSwapBytes) swapbytes(t, order);
		UV->t[i] = 86400.0*(t[0]+t[1]);
	}

	return 1;
}

void _UVFitsconstructor(UVFits UV)
{
	_Fitsconstructor((Fits)UV);	/* First line of _UVFitsconstructor */
	UV->fqtab.nfreq = 0;
	UV->fqtab.freq = 0;
	UV->fqtab.bw = 0;
	UV->fqtab.cw = 0;
	UV->fqtab.sideband = 0;
	UV->antab.nant = 0;
	UV->antab.name = 0;
	UV->t = 0;
	UV->reclen = 0;
}

void _UVFitsdestructor(UVFits UV)
{
	if(UV->fqtab.freq) g_free(UV->fqtab.freq);
	if(UV->fqtab.bw) g_free(UV->fqtab.cw);
	if(UV->fqtab.cw) g_free(UV->fqtab.bw);
	if(UV->fqtab.sideband) g_free(UV->fqtab.sideband);
	if(UV->t) deleteVector(UV->t);

	_Fitsdestructor((Fits)UV);	/* Last line of _UVFitsdestructor */
}

UVFits newUVFits()
{
	UVFits UV;

	UV = g_new(struct _UVFits, 1);
	_UVFitsconstructor(UV);

	return UV;
}

void deleteUVFits(UVFits UV)
{
	if(!UV) return;

	_UVFitsdestructor(UV);

	g_free(UV);
}

UVFits openUVFits(const char *filename)
{
	UVFits UV;
	FitsTable T;
	int ret, axislen, i, n;

	UV = newUVFits();

	if(_Fitsopen((Fits)UV, filename) == 0)
	{
		deleteUVFits(UV);
		return 0;
	}

	UV->reclen = 1;
	T = ((Fits)UV)->tables[0];
	for(i = 0; i < T->naxes; i++)
	{
		n = T->axes[i].n;
		if(n > 0) UV->reclen *= n;
	}
	UV->reclen += T->pcount;

	ret = FitsTablegetaxisinfo(((Fits)UV)->tables[0], 
		"COMPLEX", &axislen, 0, 0, 0, 0);
	g_assert(ret == 1);
	g_assert(axislen == 3);
	ret = FitsTablegetaxisinfo(((Fits)UV)->tables[0], 
		"STOKES", &axislen, 0, 0, 0, 0);
	g_assert(ret == 1);
	g_assert(axislen == 1);

	g_assert(((Fits)UV)->tables[0]->bitpix == -32);
		
	ret = UVFitsparseFQtable(UV);
	g_assert(ret == 1);

	ret = UVFitsparseANtable(UV);
	g_assert(ret == 1);

	fprintf(stderr, "UVFits reclen = %d\n", UV->reclen);

	return UV;
}
