#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include "pdf.h"

PDF newPDF(int nsamp)
{
	PDF p;

	p = g_new(struct _PDF, 1);
	p->pdf = newVector(nsamp);
	p->weight = 1.0;
	p->name = 0;
	return p;
}

void deletePDF(PDF p)
{
	g_assert(p != 0);

	deleteVector(p->pdf);
	g_free(p);
}

void blankPDF(PDF p)
{
	int i;

	g_assert(p != 0);
	g_assert(p->pdf);

	for(i = 0; i < PDFsamples(p); i++) p->pdf[i] = 0.0;	
}

/* Increment a PDF with antialiasing */
void incPDF(PDF p, double v)
{
	double index, f;	
	int i;

	g_assert(p != 0);
	g_assert(p->pdf);

	index = (v-p->v0)/p->dv;
	i = (int)index;
	f = index-i;
	if(index > 0 && index < PDFsamples(p)-1)
	{
		p->pdf[i]   += (1.0-f);
		p->pdf[i+1] += f;
	}
}

void savePDF(const PDF p, const char *filename)
{
	int i;
	double dist;
	double sum = 0.0;
	FILE *out;

	g_assert(p != 0);

	if(strcmp(filename, "-") == 0) out = stdout;
	else out = fopen(filename, "w");

	fprintf(out, "# %f %d\n", p->v0, PDFsamples(p));

	for(i = 0, dist = p->v0; i < PDFsamples(p); i++, dist += p->dv)
	{
		fprintf(out, "%f %e\n", dist, p->pdf[i]);
		sum += p->pdf[i]*p->dv;
	}

	if(strcmp(filename, "-") != 0) fclose(out);
}

PDF loadPDF(const char *filename)
{
	PDF p;
	FILE *in;
	int nsamp;
	char c;
	int i;
	double v0, d, f;

	if(strcmp(filename, "-") == 0) in = stdin;
	else in = fopen(filename, "r");

	if(!in) return 0;

	if(fscanf(in, "%c%lf%d", &c, &v0, &nsamp) != 3)
	{
		fprintf(stderr, "Weird -- bad file in loadPDF : %s\n",
			filename);
		if(strcmp(filename, "-") != 0) fclose(in);
		return 0;
	}

	p = newPDF(nsamp);
	p->v0 = v0;
	for(i = 0; i < nsamp; i++)
		if(fscanf(in, "%lf%lf", &d, &f) != 2)
		{
			fprintf(stderr, 
				"Weird -- bad line %d in loadPDF : %s\n", i,
					filename);
			if(strcmp(filename, "-") != 0) fclose(in);
			return 0;
		}
		else p->pdf[i] = f;

	p->dv = (d-v0)/(nsamp-1.0);

	if(strcmp(filename, "-") != 0) fclose(in);

	return p;
}

double PDFtotalarea(const PDF p)
{
	double sum=0.0;
	int i;

	for(i = 0; i < PDFsamples(p); i++) sum+=p->pdf[i];
	return sum*p->dv;
}

/* Get area of PDF bounded by v1 and v2, with interpolation 
 * return 0.0 if v2 < v1 
 */
double PDFarea(const PDF p, double v1, double v2)
{
	double area = 0.0;
	double f1, f2;
	int i1, i2, i;

	if(v1 < p->v0 - 0.5*p->dv)            
		v1 = p->v0 - 0.5*p->dv;
	if(v1 > p->v0 + (PDFsamples(p)-0.5)*p->dv) 
		v1 = p->v0 - (PDFsamples(p)-0.5)*p->dv;
	if(v2 < p->v0 - 0.5*p->dv)            
		v2 = p->v0 - 0.5*p->dv;
	if(v2 > p->v0 + (PDFsamples(p)-0.5)*p->dv) 
		v2 = p->v0 - (PDFsamples(p)-0.5)*p->dv;

	if(v2 <= v1) return 0.0;

	f1 = (v1-p->v0)/p->dv;
	f2 = (v2-p->v0)/p->dv;
	i1 = (int)(f1 + 0.5);
	i2 = (int)(f2 + 0.5);
	if(i1 < 0) i1 = 0;
	if(i2 > PDFsamples(p)-1) i2 = PDFsamples(p)-1;

	/* Subtract amount lost from endpoints */

	area -= (f1 - i1 - 0.5)*p->pdf[i1];
	area -= (i2 + 0.5 - f2)*p->pdf[i2];

	for(i = i1; i <= i2; i++) area += p->pdf[i];

	area *= p->dv;
	
	return area;
}

double PDFpeak(const PDF p, double v1, double v2)
{
	double best;
	int i1, i2, i, peak;
	
	g_assert(v2 > v1);

	i1 = (v1-p->v0)/p->dv;
	i2 = (v2-p->v0)/p->dv;

	if(i1 < 0) i1 = 0;
	if(i1 > PDFsamples(p)-1) i1 = PDFsamples(p)-1;
	if(i2 < 0) i2 = 0;
	if(i2 > PDFsamples(p)-1) i2 = PDFsamples(p)-1;
	g_assert(i2 > i1);

	best = p->pdf[i2];
	peak = i2;

	for(i = i1; i < i2; i++) if(p->pdf[i] > best)
	{
		best = p->pdf[i];
		peak = i;
	}

	return p->v0 + peak*p->dv;
}

double PDFsigmarange(const PDF p, double nsigma, double v1, double v2,
	double *x1, double *x2)
{
	int b1, b2, bsize;
	int i1, i2, j1, j2;
	double sum;
	double goal;

	goal = erf(nsigma/sqrt(2.0));

	goal /= p->dv;

	if(v2 <= v1) return -1.0;

	i1 = (v1-p->v0)/p->dv;
	i2 = (v2-p->v0)/p->dv;

	if(i1 < 0) i1 = 0;
	if(i1 > PDFsamples(p)-1) i1 = PDFsamples(p)-1;
	if(i2 < 0) i2 = 0;
	if(i2 > PDFsamples(p)-1) i2 = PDFsamples(p)-1;
	if(i2 <= i1) return -1.0;

	bsize = PDFsamples(p)*2;
	b1 = b2 = -1;
	j1 = j2 = i1;
	sum = p->pdf[i1];

	do
	{
		if(sum < goal || j1 >= j2)
		{
			j2++;
			if(j2 > i2) break;
			sum += p->pdf[j2];
			if(sum >= goal && j2-j1 < bsize)
			{
				b1 = j1;
				b2 = j2;
				bsize = b2-b1;
			}
		}
		else
		{
			sum -= p->pdf[j1];
			if(sum > goal && j2-j1 < bsize)
			{
				b1 = j1;
				b2 = j2;
				bsize = b2-b1;
			}
			j1++;
		}
	}
	while(j2 <= i2);

	if(b1 < 0) return -1.0;

	*x1 = p->v0 + p->dv*b1;
	*x2 = p->v0 + p->dv*b2;

	return *x2 - *x1;
}

void scalePDF(PDF p, double factor)
{
	int i;

	g_assert(p != 0);
	
	for(i = 0; i < PDFsamples(p); i++) p->pdf[i] *= factor;
}

void PDFaddscalar(PDF p, double scalar)
{
	int i;

	g_assert(p != 0);
	
	for(i = 0; i < PDFsamples(p); i++) p->pdf[i] += scalar;
}

/* applies a double func(double) to each element */
void PDFapplyfunction(PDF p, double (*func)(double x))
{
	int i;

	g_assert(p != 0);
	
	for(i = 0; i < PDFsamples(p); i++) p->pdf[i] = func(p->pdf[i]);
}

/* Normalize such that Int(pdf(v)*dv) = 1.0 */
/* Returns initial sum */
double normPDF(PDF p)
{
	int i;
	double sum = 0.0;
	
	g_assert(p != 0);
	
	for(i = 0; i < PDFsamples(p); i++) sum += p->pdf[i];
	if(sum != 0.0) scalePDF(p, 1.0/(sum*p->dv));
	return sum*p->dv;
}

/* Integrate until we get frac.  return dist. */
double PDFgetmin(const PDF p, double frac)
{
	int i;
	double dist;
	double f = 0.0;

	g_assert(p != 0);
	
	for(i = 0, dist = p->v0; i < PDFsamples(p); i++, dist += p->dv)
	{
		f += p->dv*p->pdf[i];
		if(f > frac) return dist;
	}
	return -1.0;
}

/* Same as above, but integrate backwards from other end. */
double PDFgetmax(const PDF p, double frac)
{
	int i;
	double dist, d1;
	double f = 0.0;

	g_assert(p != 0);
	
	d1 = p->v0 + (PDFsamples(p)-1.0)*p->dv;

	for(i = PDFsamples(p)-1, dist = d1; i >= 0; i--, dist -= p->dv)
	{
		f += p->dv*p->pdf[i];
		if(f > frac) return dist;
	}
	return -1.0;
}

/* Simple linear interpolation of the pdf */
/* Returns 0 outside range */
double interpolatePDF(const PDF p, double v)
{
	int index;
	double a, b, f, k;

	g_assert(p != 0);

	index = (int)((v - p->v0)/p->dv);
	if(index < 0 || index >= PDFsamples(p)) a = 0.0;
	else a = p->pdf[index];

	index++;
	if(index < 0 || index >= PDFsamples(p)) b = 0.0;
	else b = p->pdf[index];

	f = (double)index - (v - p->v0)/p->dv;

	k = f*a + (1.0-f)*b;

	if(isnan(k))
	{
		fprintf(stderr, "NAN! interpolatePDF @ %f, %f %f %d\n",
			v, p->v0, p->dv, PDFsamples(p));
	}

	return k;
}

double PDFminval(const PDF p)
{
	int index;
	double min;
	
	g_assert(p != 0);
	
	min = p->pdf[0];

	for(index = 1; index < PDFsamples(p); index++)
		if(p->pdf[index] < min) min = p->pdf[index];

	return min;
}

double PDFmaxval(const PDF p)
{
	int index;
	double max;
	
	g_assert(p != 0);
	
	max = p->pdf[0];

	for(index = 1; index < PDFsamples(p); index++)
		if(p->pdf[index] > max) max = p->pdf[index];

	return max;
}


