#include "lu.h"
#include <glib.h>
#include <math.h>

intVector ludecompose(Matrix a)
{
	int n, i, imax, j, k;
	double big, dum, sum, temp;
	Vector vv;
	intVector indx;
	const double TINY=1.0e-30;
	int d = 1;

	g_assert(a);
	g_assert(Matrixissquare(a));

	n = MatrixSize1(a);

	vv = newVector(n);

	/* allocate an extra byte and store number of permutations after */
	indx = newintVector(n+1);
	VectorSize(indx) = n;	
	
	for(i = 0; i < n; i++)
	{
		big = 0.0;
		for(j = 0; j < n; j++)
			if((temp = fabs(a[i][j])) > big) big=temp;
		g_assert(big > 0.0);
		vv[i] = 1.0/big;
	}
	for(j = 0; j < n; j++)
	{
		for(i = 0; i < j; i++)
		{
			sum = a[i][j];
			for(k = 0; k < i; k++) sum -= a[i][k]*a[k][j];
			a[i][j] = sum;
		}
		big = 0.0;
		imax = 0;
		for(i = j; i < n; i++)
		{
			sum = a[i][j];
			for(k = 0; k < j; k++) sum -= a[i][k]*a[k][j];
			a[i][j] = sum;
			if((dum=vv[i]*fabs(sum)) >= big)
			{
				big = dum;
				imax = i;
			}
		}
		if(j != imax)
		{
			for(k = 0; k < n; k++)
			{
				dum = a[imax][k];
				a[imax][k] = a[j][k];
				a[j][k] = dum;
			}
			d = -d;
			dum = vv[imax];
			vv[imax] = vv[j];
			vv[j] = dum;
		}
		indx[j] = imax;
		if(a[j][j] == 0.0) a[j][j] = TINY;
		if(j < n-1)
		{
			dum = 1.0/a[j][j];
			for(i = j+1; i < n; i++) a[i][j] *= dum;
		}
	}

	deleteVector(vv);
	indx[n] = d;

	return indx;
}

Vector lubacksub(const Matrix a, const intVector indx, const Vector B)
{
	Vector b;
	int n, i, ii=-1, ip, j;
	double sum;

	g_assert(a);
	g_assert(Matrixissquare(a));

	n = MatrixSize1(a);

	g_assert(VectorSize(B) == n);

	b = dupVector(B);

	for(i = 0; i < n; i++)
	{
		ip = indx[i];
		sum = b[ip];
		b[ip] = b[i];
		if(ii >= 0) for(j = ii; j < i; j++) sum -= a[i][j]*b[j];
		else if(sum) ii = i;
		b[i] = sum;
	}
	for(i = n-1; i >= 0; i--)
	{
		sum = b[i];
		if(i < n-1) for(j = i+1; j < n; j++) sum -= a[i][j]*b[j];
		b[i] = sum/a[i][i];
	}

	return b;
}

Matrix luinverse(const Matrix M)
{
	Matrix a, b;
	Vector c, d;
	int i, j, n;
	intVector index;
	
	g_assert(M);
	g_assert(Matrixissquare(M));
	
	n = MatrixSize1(M);
	
	a = dupMatrix(M);
	index = ludecompose(a);

	b = newMatrix(n, n);
	c = newVector(n);
	zeroVector(c);
	
	for(j = 0; j < n; j++)
	{
		c[j] = 1.0;
		d = lubacksub(a, index, c);
		for(i = 0; i < n; i++) b[i][j] = d[i];
		deleteVector(d);
		c[j] = 0.0;
	}

	deleteMatrix(a);
	deleteVector(c);
	deleteintVector(index);

	return b;
}

double Matrixdeterminant(const Matrix M)
{
	Matrix a;
	intVector index;
	int i, n;
	double det = 1.0;
	
	g_assert(M);
	g_assert(Matrixissquare(M));
	
	n = MatrixSize1(M);
	
	a = dupMatrix(M);
	index = ludecompose(a);

	for(i = 0; i < n; i++) det *= a[i][i];

	if(index[n] < 0) det = -det;
	
	deleteintVector(index);
	deleteMatrix(a);

	return det;
}

/* compute A^-1 B  A must be square, and MatrixSize1(B) must match A */
Matrix Matrixinvertmultiply(const Matrix A, const Matrix B)
{
	Matrix a, b;
	Vector c, d;
	int i, j, n, m;
	intVector index;
	
	g_assert(A);
	g_assert(Matrixissquare(A));
	
	n = MatrixSize1(A);

	g_assert(MatrixSize1(B) == n);
	m = MatrixSize2(B);
	
	a = dupMatrix(A);
	index = ludecompose(a);

	b = newMatrix(n, m);
	c = newVector(n);
	zeroVector(c);
	
	for(j = 0; j < m; j++)
	{
		for(i = 0; i < n; i++) c[i] = B[i][j];
		d = lubacksub(a, index, c);
		for(i = 0; i < n; i++) b[i][j] = d[i];
		deleteVector(d);
	}

	deleteMatrix(a);
	deleteVector(c);
	deleteintVector(index);

	return b;
}

/* computes Moore-Penrose generalized matrix inverse */
Matrix Matrixpseudoinverse(const Matrix M)
{
	Matrix Mt, MtM, PsI;
	
	g_assert(M);

	Mt = transposeMatrix(M);
	MtM = Matrixmultiply(Mt, M);
	PsI = Matrixinvertmultiply(MtM, Mt);

	deleteMatrix(Mt);
	deleteMatrix(MtM);

	return PsI;
}
