#include <stdio.h>
#include <math.h>
#include <glib.h>
#include "vector.h"
#include "intvector.h"
#include "vecarray.h"

#include "lapack.h"

#include "lapack-private.h"

int MatrixLUinvert(Matrix M)
{
	int N, info, lwork;
	intVector pivot;
	Vector work;

	if(!M) return 0;

	N = MatrixSize1(M);

	g_assert(MatrixSize2(M) == N);
	g_assert(Matrixstride(M) == N);

	lwork = 16*N;

	work = newVector(lwork);
	pivot = newintVector(N);

	dgetrf_(&N, &N, M[0], &N, pivot, &info);
	if(info != 0) 
	{
		deleteVector(work);	
		deleteintVector(pivot);
		return 0;
	}

	dgetri_(&N, M[0], &N, pivot, work, &lwork, &info);
	if(info != 0) 
	{
		deleteVector(work);	
		deleteintVector(pivot);
		return 0;
	}

	deleteintVector(pivot);
	deleteVector(work);

	return 1;
}

Matrix MatrixLUinverse(const Matrix M)
{
	Matrix N;

	if(!M) return 0;

	N = dupMatrix(M);
	if(!MatrixLUinvert(N))
	{
		deleteMatrix(N);
		return 0;
	}

	return N;
}

ZMatrix ZMatrixLUdecomp(ZMatrix M)
{
	return 0;
}

int ZMatrixLUinvert(ZMatrix M)
{
	int N, info, lwork;
	intVector pivot;
	ZVector work;

	if(!M) return 0;

	N = ZMatrixSize1(M);
	g_assert(ZMatrixSize2(M) == N);
	g_assert(ZMatrixstride(M) == N);

	transposeZMatrixinplace(M);

	lwork = 16*N;

	work = newZVector(lwork);
	pivot = newintVector(N);

	zgetrf_(&N, &N, M[0], &N, pivot, &info);
	if(info != 0)
	{
		deleteZVector(work);	
		deleteintVector(pivot);
		return 0;
	}

	zgetri_(&N, M[0], &N, pivot, work, &lwork, &info);
	if(info != 0) 
	{
		deleteZVector(work);	
		deleteintVector(pivot);
		return 0;
	}

	deleteintVector(pivot);
	deleteZVector(work);

	transposeZMatrixinplace(M);

	return 1;
}

ZMatrix ZMatrixLUinverse(const ZMatrix M)
{
	ZMatrix N;

	if(!M) return 0;

	N = dupZMatrix(M);
	if(!ZMatrixLUinvert(N))
	{
		deleteZMatrix(N);
		return 0;
	}

	return N;
}

/* set upper to 1 for the upper matrix.  otherwise lower is returned */
ZMatrix ZMatrixCholeskydecomp(ZMatrix M, int upper)
{
	int n, info;
	ZMatrix A;
	char uplo = 'U';
	int i, j;

	g_assert(M);
	n = ZMatrixSize1(M);
	g_assert(n == ZMatrixSize2(M));

	A = dupZMatrix(M);
	if(upper) uplo = 'L';	/* use the opposite convention for matrix
				   reordering */

	zpotrf_(&uplo, &n, A[0], &n, &info);
	if(info != 0) fprintf(stderr, "Warning, ZMatrixCholeskydecomp: "
		"zpotrf_ returned with exit value %d\n", info);

	if(upper) for(i = 1; i < n; i++) for(j = 0; j < i; j++)
		A[i][j].re = A[i][j].im = 0.0;
	else for(i = 1; i < n; i++) for(j = 0; j < i; j++)
		A[j][i].re = A[j][i].im = 0.0;

	return A;
}

int ZMatrixCholeskyinvert(ZMatrix M)
{
	int n, i, j, info;
	char uplo = 'L';

	if(!M) return 0;

	n = ZMatrixSize1(M);
	g_assert(n == ZMatrixSize2(M));

	zpotrf_(&uplo, &n, M[0], &n, &info);
	if(info != 0) return 0;

	zpotri_(&uplo, &n, M[0], &n, &info);
	if(info != 0) return 0;

	else for(i = 1; i < n; i++) for(j = 0; j < i; j++)
	{
		M[i][j].re =  M[j][i].re;
		M[i][j].im = -M[j][i].im;
	}

	return 1;
}

int triZMatrixinvert(ZMatrix M, int upper)
{
	int n, info, i, j;
	char uplo = 'U';
	char diag = 'N';

	g_assert(M);
	n = ZMatrixSize1(M);
	g_assert(n == ZMatrixSize2(M));
	
	if(upper) uplo = 'L';
	
	ztrtri_(&uplo, &diag, &n, M[0], &n, &info);
	if(info != 0) return 0;

	if(upper) for(i = 1; i < n; i++) for(j = 0; j < i; j++)
		M[i][j].re = M[i][j].im = 0.0;
	else for(i = 1; i < n; i++) for(j = 0; j < i; j++)
		M[j][i].re = M[j][i].im = 0.0;

	return 1;
}

/* compute eigenvalues and optionally eigenvectors of Hermitian matrix */
/* eigenvectors end up as rows in M if vecs != 0 */
Vector ZMatrixEigenH(ZMatrix M, int vecs)
{
	int i, j, n, info, lwork;
	char jobz;
	char uplo = 'L';
	double dw=0.0;
	ZMatrix A;
	ZVector work=0;
	Vector rwork, v;

	if(!M) return 0;
	
	n = ZMatrixSize1(M);
	g_assert(n == ZMatrixSize2(M));

	if(vecs)
	{
		A = M;
		jobz = 'V';
	}
	else
	{
		A = dupZMatrix(M);
		jobz = 'N';
	}

	v = newVector(n);
	rwork = newVector(3*n-2);

	lwork = -1;
	zheev_(&jobz, &uplo, &n, A[0], &n, v, (ZVector)(&dw), 
		&lwork, rwork, &info);
	if(info != 0) fprintf(stderr, "Warning, : ZMatrixEigenH : "
		"zheev_ (1) returned with exit value %d\n", info);

	lwork = dw;
	work = newZVector(lwork);

	/* FIXME : should I really need to reset these values? */
	if(vecs) jobz = 'V'; else jobz = 'N';
	uplo = 'L';

	zheev_(&jobz, &uplo, &n, A[0], &n, v, work, &lwork, rwork, &info);
	if(info != 0) fprintf(stderr, "Warning, : ZMatrixEigenH : "
		"zheev_ (2) returned with exit value %d\n", info);
	
	deleteVector(rwork);
	deleteZVector(work);

	if(vecs) /* due to fortran <--> c conversions, need to conjugate vecs */
	{
		for(j = 0; j < n; j++) for(i = 0; i < n; i++)
			M[j][i].im = -M[j][i].im;
	}
	else deleteZMatrix(A);

	return v;
}


/* move to svd.c? */

int nonsymeigensolve(const Matrix V, Vector REvals,
	Vector IMvals, Matrix vecs)
{
	Matrix Vt;
	Vector work;
	int N, lwork, info;
	char *calcvecs;

	g_assert(V);
	N = MatrixSize1(V);
	g_assert(MatrixSize2(V) == N);
	g_assert(VectorSize(REvals) == N);
	g_assert(VectorSize(IMvals) == N);
	g_assert(MatrixSize1(vecs) == N);
	g_assert(MatrixSize2(vecs) == N);

	Vt = transposeMatrix(V);

	lwork = 8*N;
	work = newVector(lwork);

	if(vecs) calcvecs = "V"; else calcvecs = "N";

	dgeev_("N", calcvecs, &N, Vt[0], &N, REvals, IMvals, 
		vecs[0], &N, vecs[0], &N,
		work, &lwork, &info);

	deleteVector(work);
	deleteMatrix(Vt);

	if(info != 0) return 0;

	if(vecs) transposeMatrixinplace(vecs);

	return 1;
}

int expMatrix(Matrix M, double factor)
{
	int N, i, j, ok;
	Vector REvals, IMvals;
	Matrix EV, EVi;
	double f;	

	if(!M) return 0;
	N = MatrixSize1(M);
	g_assert(MatrixSize2(M) == N);

	REvals = newVector(N);
	IMvals = newVector(N);
	EV = newMatrix(N, N);

	ok = nonsymeigensolve(M, REvals, IMvals, EV);
	
	if(!ok)
	{
		deleteMatrix(EV);
		deleteVector(REvals);
		deleteVector(IMvals);

		return 0;
	}

	EVi = MatrixLUinverse(EV);
	if(!EVi)
	{
		deleteMatrix(EV);
		deleteVector(REvals);
		deleteVector(IMvals);

		return 0;
	}

	for(j = 0; j < N; j++) 
	{
		f = exp(factor*REvals[j]);
		for(i = 0; i < N; i++) EVi[j][i] *= f;
	}

	copyMatrixmultiply(M, EV, EVi);

	deleteMatrix(EV);
	deleteMatrix(EVi);
	deleteVector(REvals);
	deleteVector(IMvals);

	return 1;
}

int powMatrix(Matrix M, double power)
{
	int N, i, j, ok;
	Vector REvals, IMvals;
	Matrix EV, EVi;
	double f;	

	if(!M) return 0;
	
	N = MatrixSize1(M);
	g_assert(MatrixSize2(M) == N);

	REvals = newVector(N);
	IMvals = newVector(N);
	EV = newMatrix(N, N);

	ok = nonsymeigensolve(M, REvals, IMvals, EV);

	if(!ok)
	{
		deleteMatrix(EV);
		deleteVector(REvals);
		deleteVector(IMvals);

		return 0;
	}
	
	EVi = MatrixLUinverse(EV);
	if(!EVi)
	{
		deleteMatrix(EV);
		deleteVector(REvals);
		deleteVector(IMvals);

		return 0;
	}

	for(j = 0; j < N; j++) 
	{
		f = pow(REvals[j], power);
		for(i = 0; i < N; i++) EVi[j][i] *= f;
	}

	copyMatrixmultiply(M, EV, EVi);

	deleteMatrix(EV);
	deleteMatrix(EVi);
	deleteVector(REvals);
	deleteVector(IMvals);

	return 1;
}

/* computes Moore-Penrose generalized matrix inverse */
Matrix Matrixpseudoinverse(const Matrix M)
{
	Matrix Mt, MtM, PsI;
	
	if(!M) return 0;

	Mt = transposeMatrix(M);
	MtM = Matrixmultiply(Mt, M);
	MatrixLUinvert(MtM);
	PsI = Matrixmultiply(MtM, Mt);

	deleteMatrix(Mt);
	deleteMatrix(MtM);

	return PsI;
}
