#include <glib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include "fitsbase.h"
#include "byteorder.h"

/* TODO:
 * 
 * add "ownsdata" to tables.
 * if table owns data, then delete it when deleted.
 */

int FitsSwapBytes = 1;

void Fitsinit()
{
	g_assert(sizeof(double) == 8);
	g_assert(sizeof(float) == 4);
	g_assert(sizeof(int) == 4);
	g_assert(sizeof(char) == 1);
	if(byteorder() == BO_INTEL) 
	{
		FitsSwapBytes = 1;
		fprintf(stderr, "Need to swap bytes\n");
	}
	else FitsSwapBytes = 0;
}

int FitsTabletypeoffset(const FitsTable T, const char *type, char form,int *num)
{
	int i, off = 0, n, s;
	
	g_assert(T);
	g_assert(T->axes);
	g_assert(T->columns);
	g_assert(abs(T->bitpix) == 8);
	
	for(i = 0; i < T->tfields; i++)
	{
		n = T->columns[i].tformat[0] - '0';
		if(strncmp(type, T->columns[i].ttype, strlen(type)) == 0) 
		{
			if(T->columns[i].tformat[1] != form) return -1;
			if(num) *num = n;
			return off;
		}
		switch(T->columns[i].tformat[1])
		{
			case 'A' : s = 1 ; break;
			case 'J' : s = 4 ; break;
			case 'E' : s = 4 ; break;
			case 'D' : s = 8 ; break;
			default: return -1;
		}
		off += n*s;
	}

	return -1;
}

double *FitsTablegetdatadouble(const FitsTable T, const char *type, int row,
	int *num)
{
	double *d;
	int off, len, n;
	int order[][2] = {{0, sizeof(double)}, {0, 0}};

	off = FitsTabletypeoffset(T, type, 'D', &n);
	g_assert(off >= 0);

	len = T->axes[0].n;
	
	d = g_new(double, n);

	memcpy(d, T->data+len*row+off, 8*n);

	if(num) *num = n;

	if(FitsSwapBytes) 
	{
		order[0][0] = n;
		swapbytes(d, order);
	}

	return d;
}

float *FitsTablegetdatafloat(const FitsTable T, const char *type, int row,
	int *num)
{
	float *d;
	int off, len, n;
	int order[][2] = {{0, sizeof(float)}, {0, 0}};

	off = FitsTabletypeoffset(T, type, 'E', &n);
	g_assert(off >= 0);

	len = T->axes[0].n;
	
	d = g_new(float, n);

	memcpy(d, T->data+len*row+off, 4*n);

	if(num) *num = n;

	if(FitsSwapBytes) 
	{
		order[0][0] = n;
		swapbytes(d, order);
	}

	return d;
}

int *FitsTablegetdataint(const FitsTable T, const char *type, int row,
	int *num)
{
	int *d;
	int off, len, n;
	int order[][2] = {{0, sizeof(int)}, {0, 0}};

	off = FitsTabletypeoffset(T, type, 'J', &n);
	g_assert(off >= 0);

	len = T->axes[0].n;
	
	d = g_new(int, n);

	memcpy(d, T->data+len*row+off, 4*n);

	if(num) *num = n;
	
	if(FitsSwapBytes) 
	{
		order[0][0] = n;
		swapbytes(d, order);
	}

	return d;
}

char *FitsTablegetdatachar(const FitsTable T, const char *type, int row,
	int *num)
{
	char *d;
	int off, len, n;

	off = FitsTabletypeoffset(T, type, 'A', &n);
	g_assert(off >= 0);

	len = T->axes[0].n;
	
	d = g_new(char, n+1);

	memcpy(d, T->data+len*row+off, n);
	d[n] = 0;

	if(num) *num = n;

	return d;
}


FitsTable newFitsTable()
{
	FitsTable T;

	T = g_new(struct _FitsTable, 1);

	T->kv = newKeyValue();
	T->base = 0;
	T->data = 0;
	T->length = 0;
	T->name = 0;
	T->extver = 0;
	T->naxes = 0;
	T->axes = 0;
	T->bitpix = 0;
	T->pcount = 0;
	T->gcount = 0;
	T->pdata = 0;
	T->tfields = 0;
	T->columns = 0;

	return T;
}

FitsTable newFitsTablefromdata(char *data)
{
	FitsTable T;
	int i, res;
	char line[81];
	const char *name;

	fprintf(stderr, "tablestart = %p\n", data);

	if(strncmp(data, "SIMPLE  ", 8) != 0 && 
	   strncmp(data, "XTENSION", 8) != 0)
	{
		fprintf(stderr, "Table header bad???\n");
		return 0;
	}
	
	T = newFitsTable();
	T->base = data;
	
	for(i = 0; ; i++)
	{
		strncpy(line, data+80*i, 80);
		line[80] = 0;
		
		if(line[8] == '=')
			FitsTableaddkv(T, line);

		if(strncmp(line, "END", 3) == 0) break;
	}

	i += 36;
	i -= (i % 36);

	fprintf(stderr, "header size = %x\n", 80*i);

	T->data = data+80*i;
	
//	printKeyValue(T->kv);
	
	T->bitpix = getKeyValueint(T->kv, "BITPIX");
	g_assert(T->bitpix != KV_INTERR);
	g_assert(abs(T->bitpix) % 8 == 0);

	name = getKeyValuestring(T->kv, "EXTNAME");
	if(name) T->name = g_strdup(name);

	T->extver = getKeyValueint(T->kv, "EXTVER");
	if(T->extver == KV_INTERR) T->extver = 0;
	
	res = FitsTablegetaxes(T);
	g_assert(res > 0);

	fprintf(stderr, "tablelength = %x\n", T->length);
	
	return T;
}

void deleteFitsTable(FitsTable T)
{
	if(!T) return;

	if(T->kv) deleteKeyValue(T->kv);
	if(T->name) g_free(T->name);
	if(T->axes) g_free(T->axes);
	if(T->pdata) g_free(T->pdata);
	if(T->columns) g_free(T->columns);

	g_free(T);
}

/* Warning -- this destroys line[].  Line must be >=80 chars */
int FitsTableaddkv(FitsTable T, char *line)
{
	int i;
	int q1 = 0, q2 = 0;
	char key[100], value[100];
	
	for(i = 0; i < 80; i++)
	{
		if(line[i] == '/')
		{
			if(i < 10) return 0;
			line[i] = 0;
			break;
		}
		if(line[i] == '\'')
		{
			if(i < 9) return 0;
			if(q1 == 0) q1 = i;
			else if(q2 == 0) 
			{
				q2 = i;
				line[i] = 0;
			}
			else return 0;
		}
	}
	if(q1 > 0 && q2 == 0) return 0;
	line[8] = 0;
	sscanf(line, "%s", key);
	if(q1 == 0) 
	{
		sscanf(line+9, "%s", value);
		KeyValueaddparm(T->kv, key, value);
	}
	else KeyValueaddparm(T->kv, key, line+q1+1);

	return 1;
}

int FitsTablegetaxes(FitsTable T)
{
	char key[100];
	int i, val;
	int len = 1;
	double d;

	g_assert(T);
	g_assert(T->kv);

	val = FitsTablegetcolumns(T);
	g_assert(val > 0);

	val = FitsTablegetpdata(T);
	g_assert(val > 0);
	
	if(T->axes) g_free(T->axes);
	
	T->naxes = getKeyValueint(T->kv, "NAXIS");
	if(T->naxes == KV_INTERR)
	{
		T->naxes = 0;
		return 0;
	}	
	T->axes = g_new(struct axis, T->naxes);
	for(i = 0; i < T->naxes; i++)
	{
		sprintf(key, "NAXIS%d", i+1);
		val = getKeyValueint(T->kv, key);
		if(val == KV_INTERR)
		{
			T->naxes = 0;
			return 0;
		}
		T->axes[i].n = val;
		if(val > 0 || T->tfields > 0) len *= val;
		
		sprintf(key, "CTYPE%d", i+1);
		T->axes[i].ctype = getKeyValuestring(T->kv, key);
		
		sprintf(key, "CRVAL%d", i+1);
		d = getKeyValuedouble(T->kv, key);
		if(d != KV_FLOATERR) T->axes[i].crval = d;
		else T->axes[i].crval = 0;

		sprintf(key, "CRPIX%d", i+1);
		d = getKeyValuedouble(T->kv, key);
		if(d != KV_FLOATERR) T->axes[i].crpix = d;
		else T->axes[i].crpix = 0;

		sprintf(key, "CDELT%d", i+1);
		d = getKeyValuedouble(T->kv, key);
		if(d != KV_FLOATERR) T->axes[i].cdelt = d;
		else T->axes[i].cdelt = 0;
		
		sprintf(key, "CROTA%d", i+1);
		d = getKeyValuedouble(T->kv, key);
		if(d != KV_FLOATERR) T->axes[i].crota = d;
		else T->axes[i].crota = 0;
	}

	T->gcount = getKeyValueint(T->kv, "GCOUNT");
	if(T->gcount == KV_INTERR) T->gcount = 1;

	len += T->pcount;
	len *= T->gcount*abs(T->bitpix/8);

	fprintf(stderr, "Table length = %d in %d axes\n", len, T->naxes);
	len += 2879;
	len -= (len % 2880);
	fprintf(stderr, "Rounded to %d\n", len);
	T->length = len;
	
	return 1;
}

int FitsTablegetcolumns(FitsTable T)
{
	int i;
	char key[100];

	g_assert(T);
	g_assert(T->kv);

	if(T->columns) g_free(T->columns);
	
	T->tfields = getKeyValueint(T->kv, "TFIELDS");
	if(T->tfields == KV_INTERR) 
	{
		T->tfields = 0;
		return 1;
	}

	T->columns = g_new(struct column, T->tfields);
	
	for(i = 0; i < T->tfields; i++)
	{
		sprintf(key, "TTYPE%d", i+1);
		T->columns[i].ttype = getKeyValuestring(T->kv, key);
		sprintf(key, "TUNIT%d", i+1);
		T->columns[i].tunit = getKeyValuestring(T->kv, key);
		sprintf(key, "TFORM%d", i+1);
		T->columns[i].tformat = getKeyValuestring(T->kv, key);
	}
	
	return 1;
}

int FitsTablegetpdata(FitsTable T)
{
	int i;
	char key[100];

	g_assert(T);
	g_assert(T->kv);
	
	if(T->pdata) g_free(T->pdata);

	T->pcount = getKeyValueint(T->kv, "PCOUNT");
	if(T->pcount == KV_INTERR) 
	{
		T->pcount = 0;
		return 1;
	}
	
	T->pdata = g_new(struct pinfo, T->pcount);

	for(i = 0; i < T->pcount; i++)
	{
		sprintf(key, "PZERO%d", i+1);
		T->pdata[i].pzero = getKeyValuedouble(T->kv, key);
		sprintf(key, "PSCAL%d", i+1);
		T->pdata[i].pscale = getKeyValuedouble(T->kv, key);
		sprintf(key, "PTYPE%d", i+1);
		T->pdata[i].ptype = getKeyValuestring(T->kv, key);
		if(T->pdata[i].pzero == KV_FLOATERR ||
		   T->pdata[i].pscale == KV_FLOATERR) return 0;
	}
	
	return 1;
}


void _Fitsconstructor(Fits F)
{
	g_assert(F);
	
	F->fd = -1;
	F->data = 0;
	F->length = 0;
	F->ntables = 0;
	F->tables = 0;
}

void _Fitsdestructor(Fits F)
{
	FitsdeleteallTables(F);
	if(F->data) munmap(F->data, F->length);
	if(F->fd >= 0) close(F->fd);
}

int _Fitsopen(Fits F, const char *filename)
{
	FitsTable T;
	int fd;
	struct stat statbuf;
	void *data;
	char *ptr;

	g_assert(F);

	fd = open(filename, O_RDONLY);
	if(fd < 0) 
	{
		fprintf(stderr, "Cannot open file %s\n", filename);
		return 0;
	}
	
	if(fstat(fd, &statbuf) < 0)
	{
		fprintf(stderr, "fstat failed\n");
		close(fd);
		return 0;
	}
	
	if((data = mmap (0, statbuf.st_size, PROT_READ, MAP_SHARED, fd, 0))
		== MAP_FAILED)
	{
		fprintf(stderr, "mmap failed\n");
		close(fd);
		return 0;
	}

	if(strncmp((char *)data, "SIMPLE  =", 9) != 0)
	{
		fprintf(stderr, "File appears not to be a Fits file\n");
		close(fd);
		return 0;
	}
	
	F->fd = fd;
	F->data = data;
	F->length = statbuf.st_size;

	fprintf(stderr, "Fits file mapped : length = %d\n", F->length);
	
	/* Look for tables */
	for(ptr = (char *)data; (ptr - (char *)data) < F->length;)
	{
		T = newFitsTablefromdata(ptr);
		if(!T) break;
		FitsappendTable(F, T);
		ptr = (char *)T->data+T->length;
	}

	return 1;
}

int FitsTablegetaxisinfo(const FitsTable T, const char *type, int *axislen,
	double *crval, double *cdelt, double *crpix, double *crota)
{
	int i;
	
	g_assert(T);
	g_assert(T->axes);

	for(i = 0; i < T->naxes; i++)
	{
		if(T->axes[i].ctype == 0) continue;
		if(strncmp(T->axes[i].ctype, type, strlen(type)) == 0)
		{
			if(axislen) *axislen = T->axes[i].n;
			if(crval) *crval = T->axes[i].crval;
			if(cdelt) *cdelt = T->axes[i].cdelt;
			if(crpix) *crpix = T->axes[i].crpix;
			if(crota) *crota = T->axes[i].crota;
			return 1;
		}
	}

	fprintf(stderr, "FitsTablegetaxisinfo : %s not found\n", type);

	return 0;
}

Fits newFits()
{
	Fits F;

	F = g_new(struct _Fits, 1);
	_Fitsconstructor(F);

	return F;
}

void deleteFits(Fits F)
{
	if(!F) return;

	_Fitsdestructor(F);
	
	g_free(F);
}

Fits openFits(const char *filename)
{
	Fits F;

	F = newFits();

	if(_Fitsopen(F, filename)) return F;
	else 
	{
		deleteFits(F);
		return 0;
	}
}

int FitsappendTable(Fits F, FitsTable T)
{
	FitsTable *tmp;
	int i;
	
	g_assert(F);
	g_assert(T);

	tmp = F->tables;

	F->tables = g_new(FitsTable, F->ntables+1);
	if(tmp) for(i = 0; i < F->ntables; i++) F->tables[i] = tmp[i];
	F->tables[F->ntables] = T;
	F->ntables++;

	if(tmp) g_free(tmp);

	fprintf(stderr, "table %d added\n", F->ntables);
	
	return F->ntables;
}

void FitsdeleteallTables(Fits F)
{
	int i;
	
	if(!F) return;

	if(F->ntables > 0) for(i = 0; i < F->ntables; i++)
		deleteFitsTable(F->tables[i]);
	g_free(F->tables);
	F->tables = 0;
	F->ntables = 0;
}

FitsTable FitsgetFitsTablebyname(Fits F, const char *name, int version)
{
	int i;

	if(!F) return 0;
	if(F->ntables == 0) return 0;
	
	for(i = 0; i < F->ntables; i++)
	{
		if(F->tables[i]->name)
		{
			if(!name) continue;
			if(version > 0) if(version != F->tables[i]->extver)
				continue;
			if(strncmp(F->tables[i]->name, name, strlen(name)) == 0)
				return F->tables[i];
		}
		else if(!name) 
		{
			if(version > 0) if(version != F->tables[i]->extver)
				continue;
			return F->tables[i];
		}
	}
		
	return 0;
}
