#include <stdio.h>
#include <math.h>
#include <glib.h>
#include "vector.h"
#include "vecarray.h"
#include "image-vector.h"
#include "randdist.h"

#include "eigen.h"

#define POWERSMOOTH     0
#define ADAPTIVESMOOTH  1

/* Note -- most of the contents of this file are from eigen-1.01a.tgz */

void dgeev_(char *jobvl, char *jobvr, int *N, double *A, int *LDA, 
	double *wr, double *wi, double *vl, int *ldvl, double *vr, 
	int *ldvr, double *work, int *lwork, int *info);

void dgetrf_(int *m, int *n, double *a, int *lda, int *ipiv, int *info);

void dgetri_(int *n, double *a, int *lda, int *ipiv, double *work,
	int *lwork, int *info);

void MatrixLUinvert(Matrix M)
{
	int N, info, lwork;
	intVector pivot;
	Vector work;

	g_assert(M);
	N = MatrixSize1(M);
	g_assert(MatrixSize2(M) == N);

	pivot = newintVector(N);
	lwork = 16*N;
	work = newVector(lwork);

	dgetrf_(&N, &N, M[0], &N, pivot, &info);
	dgetri_(&N, M[0], &N, pivot, work, &lwork, &info);

	deleteintVector(pivot);
	deleteVector(work);
}

void nonsymeigensolve(const Matrix V, Vector REvals,
	Vector IMvals, Matrix vecs)
{
	Matrix Vt;
	Vector work;
	int N, lwork, info;
	char *calcvecs;

	g_assert(V);
	N = MatrixSize1(V);
	g_assert(MatrixSize2(V) == N);
	g_assert(VectorSize(REvals) == N);
	g_assert(VectorSize(IMvals) == N);
	g_assert(MatrixSize1(vecs) == N);
	g_assert(MatrixSize2(vecs) == N);

	Vt = transposeMatrix(V);

	lwork = 8*N;
	work = newVector(lwork);

	if(vecs) calcvecs = "V"; else calcvecs = "N";

	dgeev_("N", calcvecs, &N, Vt[0], &N, REvals, IMvals, 
		vecs[0], &N, vecs[0], &N,
		work, &lwork, &info);

	printf("dgeev_ info = %d\n", info);

	deleteVector(work);
	deleteMatrix(Vt);

	if(vecs) transposeMatrixinplace(vecs);
}

void expMatrix(Matrix M, double factor)
{
	int N, i, j;
	Vector REvals, IMvals;
	Matrix EV, EVi;
	double f;	

	g_assert(M);
	N = MatrixSize1(M);
	g_assert(MatrixSize2(M) == N);

	REvals = newVector(N);
	IMvals = newVector(N);
	EV = newMatrix(N, N);

	nonsymeigensolve(M, REvals, IMvals, EV);
	
	EVi = dupMatrix(EV);
	MatrixLUinvert(EVi);

	for(j = 0; j < N; j++) 
	{
		f = exp(factor*REvals[j]);
		for(i = 0; i < N; i++) EVi[j][i] *= f;
	}

	copyMatrixmultiply(M, EV, EVi);

	deleteMatrix(EV);
	deleteMatrix(EVi);
	deleteVector(REvals);
	deleteVector(IMvals);
}

void powMatrix(Matrix M, double power)
{
	int N, i, j;
	Vector REvals, IMvals;
	Matrix EV, EVi;
	double f;	

	g_assert(M);
	N = MatrixSize1(M);
	g_assert(MatrixSize2(M) == N);

	REvals = newVector(N);
	IMvals = newVector(N);
	EV = newMatrix(N, N);

	nonsymeigensolve(M, REvals, IMvals, EV);
	
	EVi = dupMatrix(EV);
	MatrixLUinvert(EVi);

	for(j = 0; j < N; j++) 
	{
		f = pow(REvals[j], power);
		for(i = 0; i < N; i++) EVi[j][i] *= f;
	}

	copyMatrixmultiply(M, EV, EVi);

	deleteMatrix(EV);
	deleteMatrix(EVi);
	deleteVector(REvals);
	deleteVector(IMvals);
}

struct Sturm
{   
	tridiag t;
               
        int flag; 
        double upper;
        double lower;
        struct Sturm *next;
};

/* If you want to use power smoothing instead of adaptive smoothing, define
   smoothing=POWERSMOOTH
*/
const int smoothing = POWERSMOOTH;
const double EPSILON = 1e-15;

/* struct Sturm_gen generates the sturm data for a given tridiagigonal matrix,
   and puts them in the global list sturm_list, which must be later freed
   with sturm_free.
*/
static struct Sturm *newSturmList(tridiag *m)
{
	int i, size;
	struct Sturm *cursturm, *sturmlist;

	size = m->length;
	cursturm = g_new(struct Sturm, 1);
	cursturm->t.length = 1;
	cursturm->lower = 0.0;
	cursturm->upper = 0.0;
	cursturm->flag = 0;
	cursturm->t.a = m->a;
	cursturm->t.b = m->b;
	cursturm->next = 0;

	sturmlist = cursturm;
	
	for(i = 1; i < size; i++)
	{
		if(fabs(m->b[i-1]) < EPSILON) /* if the beta is zero, then
					   it is the start of a new
					   irriducible matrix */
		{
			cursturm->next = g_new(struct Sturm, 1);
			cursturm = cursturm->next;
			cursturm->t.length = 1;
			cursturm->lower = 0.0;
			cursturm->upper = 0.0;
			cursturm->flag = 0;
			cursturm->t.a = m->a + i;
			cursturm->t.b = m->b + i;
			cursturm->next = 0;
		}
		else cursturm->t.length++;
	}

	return sturmlist;
}

static void deleteSturmList(struct Sturm *sturmlist)
{
	struct Sturm *cursturm, *tmp;

	g_assert(sturmlist);
	
	cursturm = sturmlist;
	while(cursturm)
	{
		tmp = cursturm->next;
		g_free(cursturm);
		cursturm = tmp;
	}
}

static int sign_changes(struct Sturm sturm, double x)
{
	double p0 = 1.0, p1, tmp;
	int i, w=0, m;
	double smooth;

	if(smoothing == ADAPTIVESMOOTH)
	{
		smooth = (p1=(sturm.t.a[0]-x)) != 0 ? fabs(p1) : 1; 

		if(p1 < 0.0)
		{
			p1 = -1.0;
			w++;
			m = -1;
		} 
		else if(p1 > 0.0) 
		{
			p1 = 1.0;
			m = 1;
		} 
		else 
		{
			p1 = 0.0;
			m = 1;
		}

		for(i = 1; i < sturm.t.length; i++)
		{
			tmp = (sturm.t.a[i]-x)*p1
				-sturm.t.b[i-1]*sturm.t.b[i-1]*p0/smooth;
			smooth = tmp != 0.0 ? fabs(tmp) : 1.0 ;
			/* We don't do tmp*=smooth now, because we know that tmp will
			   be 1 or -1 or 0. so we only set p1 */
			p0 = p1;
			if(tmp < 0.0)
			{
				p1 = -1.0;
				if(m == 1) w++;
				m = -1;
			}
			else if (tmp > 0.0)
			{
				p1 = 1.0;
				if(m == -1) w++;
				m = 1;
			}
		       	else p1 = 0.0;
		}
	} 
	else 
	{
		smooth = fabs(x) > 1.0 ? 1.0/fabs(x) : 1.0; 
		p1=(sturm.t.a[0]-x)*smooth;
		if(p1<0)
		{
			w++;
			m = -1;
		} 
		else m = 1;
		
		for(i = 1; i < sturm.t.length; i++)
		{
			tmp = ((sturm.t.a[i]-x)*p1
				-sturm.t.b[i-1]*sturm.t.b[i-1]*p0*smooth)
				*smooth;
			p0 = p1;
			if( (p1=tmp) < 0.0)
			{
			 	if(m == 1) w++;
				m = -1;
			} 
			else if(p1 > 0.0)
			{
				if(m == -1) w++;
				m=1;
			}
		}
	}
	
	return w;
}

/* find the bounds on the roots of the characteristic polynomial of the
   irriducible tridiagonal matrix associated with spoly, which is part
   of the matrix given to sturm_gen.
*/
static void findbounds(struct Sturm *psturm)
{
	psturm->lower = -1.0;
	for(;;)
	{
		/* if we are low enough, then stop */
		if(sign_changes(*psturm, psturm->lower) == 0)
			break;
		psturm->lower *= 2.0;
	}
	psturm->upper = 1.0;
	for(;;)
	{
		/* if we are high enough, then stop */
		if(sign_changes(*psturm, psturm->upper)==psturm->t.length)
			break;
		psturm->upper *= 2.0;
	}
}

/* find the n'th root of the characteristic polynomial of the irriducible
   tridiagonal matrix associated with spoly, which is part of the matrix
   given to sturm_gen.
*/
static double findroot(struct Sturm *psturm, int n)
{
	double a, b, mid;
	const double STURMEPSILON = 1e-10;
	/* the lower bound is always under -1 if already found. if it is
	   0, as initialized in sturm_gen, then we must call findbounds
	   to find bounds
	*/
	
	if(!psturm->lower) findbounds(psturm);
	
	a = psturm->lower;
	b = psturm->upper;

	for(;;)
	{
		mid = 0.5*(a+b);
		if(fabs((mid-a)/mid) < STURMEPSILON) break;
		if(fabs((mid-b)/mid) < STURMEPSILON) break;
		if(sign_changes(*psturm,mid) >= (psturm->t.length+1-n)) b = mid;
		else a = mid;
	}
	return mid;
}

/* tsolve solves a tridiagonal, non homogenic systems of equations.
   it solves the system Tv=b, and puts the solution in v.
   note that tsolve destroys the vector b, but does not change the
   tridiag t.
*/
int tridiag_solve(tridiag *t, Vector v, Vector b)
{
	int i;

	for(i = 0; i < t->length; i++) v[i] = t->a[i];
	for(i = t->length-1; i; --i)
	{
		if(v[i] != 0.0)
		{				/* if not zero */
			v[i-1] -= t->b[i-1]*t->b[i-1]/v[i];
			b[i] /= v[i];
			b[i-1] -= t->b[i-1]*b[i];
		}
	       	else 
		{				/* if it is zero */
			printf("GOT TO ZERO CASE IN TSOLVE!\n");
			/* I have to recheck the algorithm in the zero case */
			/* i can't be 1 */
			v[--i] = 0.0;
			/* I am not sure if this is the right thing to do: */
			/* I subtract from every b before the current one
			   the value of x[i-1] (x[n-2] in the explenation I
			   wrote) times the value in the i-1'th column in
			   that row. because this is a tridiagonal matrix,
			   I have to do it only for row i-1 and i-2.
			   the value of x[i-1] is b[i]/b[i-1]. */
			b[i-1] -= t->a[i-1]*(b[i]/b[i-1]);
			b[i-2] -= t->a[i-2]*(b[i]/b[i-1]);
			/* now do the elementary operations, like in the
			   explenation, but also update the b. */
			b[i] /= t->b[i-1];
		}
	}
	/*sum=v[0]=1;*/	/* arbitary */
	if(v[0] == 0.0)
	{
		v[0] = 1;
		/*else {
		printf("ERROR: Infinite number of solutions, in tsolve.\n");
		return(0); 
		}*/
	} 
	else v[0] = b[0]/v[0];

	for(i = 1; i < t->length; i++)
	{
		if(v[i] == 0.0)
		{
			printf("GOT TO ZERO CASE IN TSOLVE!!\n");
			/* i can't be 1 */
			/*i++;
			v[i]= -t->b[i-2]/t->b[i-1]*v[i-2];*/
			v[i] = b[i+1]/t->b[i];
			i++;
			v[i] = b[i]-t->b[i-2]/t->b[i-1]*v[i-2];
		} 
		else v[i] = b[i] - t->b[i-1]/v[i]*v[i-1];
	}
	
	return 1;
}

/* harel takes as an input a irriducible tridiagonal matrix t, a known
   eigenvalue, which was previously found by findroot, and generates the
   respective eigenvector into the output vector, which should have the same
   length as the diagonal of the given irriducible matrix.

   see Numerical Recipes in C, page 394.
*/
/* HAREL_EPSILON is the wanted accuracy of the eigenvectors.
   HAREL_MATITER is the number of iterations we should allow for the reverse
                 iteration algorithm before we choose another initial vector.
   HAREL_MAXRETRYS is the number of times we will try a new initial vector.
*/
#define HAREL_EPSILON	1e-13
#define HAREL_MAXITER	60
#define HAREL_MAXRETRYS	30
/* If the eigenvalue is to accuarate, then we sometimes can't solve the
   resulting equation. So we add to the eigenvalue HAREL_CHANGE times
   that eigenvalue. HAREL_CHANGE was chosen by trying - known problems
   disappeared when choosing this value. If problems do appear then that
   value will have to be increased. Note that if eigenvalues of an irriducible
   matrix are close to one another by a small multiple of HAREL_CHANGE,
   than we may get the incorrect eigenvector...
   Increasing HAREL_CHANGE will also increase the avarage tsolves
   per eigenvector.
*/
#define HAREL_CHANGE	1e-12 /*1e-16*/
static void harel(tridiag t, double olambda, VectorTypePointer v)
{
	Vector x[2],na,tmp;
	double lambda;
	tridiag newt;	/* t - lambda*I */ 
	int i, cur=0;	/* x[cur] acts as x0, and x[!cur] acts like y */
	int haderror=0;	/* set to 1 if tsolve has returned an error */
	double d,e;
	int l;

	l = t.length;

	lambda = olambda + HAREL_CHANGE*olambda;  /* change lambda a little */
	if(fabs(lambda-olambda) < EPSILON) lambda = EPSILON; 	/* XYZ */

	x[0] = newVector(l);
	x[1] = newVector(l);
	tmp  = newVector(l);
	na   = newVector(l);

	/* set up newt = t - lambda*I */
	for(i = 0; i < l; i++) na[i] = t.a[i] - lambda;
	newt.length = l;
	newt.a = na;
	newt.b = t.b;

newvector:
	if(haderror >= HAREL_MAXRETRYS)
	{
		fprintf(stderr, "ERROR: sturm.c:harel: Giving up finding "
				"eigenvector after %d extra trys!\n", 
				HAREL_MAXRETRYS);
		fprintf(stderr, "       Contact programmer.\n");
		return;
	}
		
	for(i = 0; i < l; i++) x[cur][i] = rand_gauss();
	normalizeVector(x[cur]);

	i = HAREL_MAXITER;
	while(i--)
	{
		/* copy x[cur] to tmp, because tsolve destroys it */
		copytoVector(tmp, x[cur]);

		if(tridiag_solve(&newt, x[1-cur], x[cur]) == 0)
		{
			haderror++;
			if(haderror == 1)
			{
				fprintf(stderr, "Warning: eigen.c:harel: "
		"having problems finding eigenvector for %.16g\n", olambda);
				fprintf(stderr, "                        "
		"Getting a new random vector\n");
			}
			goto newvector;
		}
		normalizeVector(x[1-cur]);


		/* put the difference of the new and previous
		   approximations of the eigenvector in the place of
		   the previous approximation. */
		copytoVector(x[cur], tmp);
		subfromVector(x[cur], x[1-cur]);
		d = sqrt(Vectorsumsquare(x[cur]));
		copytoVector(x[cur], tmp);
		addtoVector(x[cur], x[1-cur]);
		e = sqrt(Vectorsumsquare(x[cur]));
		cur = 1-cur;

		if(d <= HAREL_EPSILON ||  e <= HAREL_EPSILON)
			break;
	}
	if(i == -1)
	{
		haderror++;
		if(haderror == 1)
			fprintf(stderr, "Getting a new random vector\n");

		goto newvector;
	}
	
	if(haderror)	/* tell the user that the error is now gone */
		fprintf(stderr, "Notice: eigen.c:harel: Now correctly found "
				"eigenvector after %d extra trys.\n", haderror);
	for(i = 0; i < l; i++) v[i] = x[cur][i];

	deleteVector(x[0]);
	deleteVector(x[1]);
	deleteVector(tmp);
	deleteVector(na);
}

/* this is the main routine of this file. as an input it takes a symmetric
   matrix, and it's corresponding size, and it finds the n biggest roots
   of m's characteristic polynomial, and puts them in the vector v, whose
   length must be n. It also finds the corresponding eigen vectors and
   puts them in a matrix which must be n rows and size columns, if vecs !=0
*/
void tridiag_eigen(tridiag *m, int n, Vector v, VecArray vecs)
{
	int size;
	int nnoflags=1, nused=0, i, j, k, j0;
	struct Sturm *s, *sturmlist;
	struct Sturm **vs;	/* for each eigenvalue in v, remember from what sturm
			   it was */
	double root;

	size = m->length;
	g_assert(VectorSize(v) == n);
	
	sturmlist = newSturmList(m);
	vs=g_new(struct Sturm *, n);

	for(i = 1; i <= n && nnoflags != 0; i++)
	{
		nnoflags = 0;
		for(s = sturmlist; s != 0; s = s->next)
		{
			if(!s->flag) 
			{
				nnoflags++;
				root = findroot(s, i);
				if(nused < n || root > v[n-1])
				{
				/* use insertion sort */
					if(n-1 < nused) j0 = n-1;
					else j0 = nused;
					for(j = j0; j >= 1; j--)
					{
						if(root > v[j-1])
						{
							v[j] = v[j-1];
							vs[j] = vs[j-1];
						} 
						else 
						{
							v[j] = root;
							vs[j] = s;
							break;
						}
					}
					if(j == 0)
					{
						v[0] = root;
						vs[0] = s;
					}
					if(nused < n) nused++;
				} 
				else s->flag = 1;
				if(s->t.length == i) s->flag = 1;
			}
		}
	}
//	printf("Eigenvals : "); printVector(v);
	if(vecs)
	{
		g_assert(VecArrayVectorSize(vecs) == size);
		g_assert(VecArraySize(vecs) == n);
		for(i = 0; i < n; i++)
		{
			k = vs[i]->t.a - m->a;
			for(j = 0; j < k; j++) vecs[i][j] = 0;
			harel(vs[i]->t, v[i], vecs[i] + k);
			for(j = k + vs[i]->t.length; j < size; j++) 
				vecs[i][j] = 0;
		}
	}
		
	g_free(vs);
	deleteSturmList(sturmlist);
}

tridiag *newtridiag(int L)
{
	tridiag *t;

	t = g_new(tridiag, 1);
	t->length = L;
	t->a = newVector(L);
	t->b = newVector(L)+1;

	return t;
}

void deletetridiag(tridiag *t)
{
	deleteVector(t->a);
	deleteVector(t->b-1);
	g_free(t);
}

tridiag *householder(Matrix a)
{
	double scale, hh, h, g, f;
	int i, j, k, l, n;
	tridiag *t;
	VectorTypePointer t0, t1;

	g_assert(Matrixissquare(a));
	t = newtridiag(MatrixSize1(a));
	t0 = t->a;
	t1 = t->b;
	n = MatrixSize1(a);
    
	for(i = n-1; i >= 1; i--) 
	{
		l = i - 1;
		h = scale = 0.0;
		if(l > 0) 
		{
			for(k = 0; k < i; k++) scale += fabs(a[i][k]);

			if(scale == 0.0) t1[l] = a[i][l];
			else 
			{
				for(k = 0; k < i; k++) 
				{
					a[i][k] /= scale;
					h += a[i][k]*a[i][k];
				}
				f = a[i][l];
				g = f > 0 ? -sqrt(h) : sqrt(h);
				t1[l] = scale*g;
				h -= f*g;
				a[i][l] = f - g;
				f = 0.0;
				for(j = 0; j < i; j++) 
				{
					a[j][i] = a[i][j]/h;
					g = 0.0;
					for(k = 0; k <= j; k++)
						g += a[j][k]*a[i][k];
					for(k = j+1 ;k <= l; k++)
						g += a[k][j]*a[i][k];
					t1[j-1] = g/h;
					f += t1[j-1]*a[i][j];
				}
				hh = 0.5*f/h;
				for(j = 0; j < i; j++) 
				{
					f = a[i][j];
					g = t1[j-1] -= hh*f;
					for(k = 0; k <= j; k++) a[j][k] -= 
						(f*t1[k-1]+g*a[i][k]);
				}
			}
		} 
		else t1[l] = a[i][l];
		t0[i] = h;
	}
	t0[0] = 0.0;
	for(i = 0; i < n; i++)
	{ 
		if(t0[i]) 
		{
			for(j = 0; j < i; j++) 
			{
				g = 0.0;
				for(k = 0; k < i; k++) g += a[i][k]*a[k][j];
				for(k = 0; k < i; k++) a[k][j] -= g*a[k][i];
			}
		}
		t0[i] = a[i][i];
		a[i][i] = 1.0;
		for(j = 0; j < i; j++) a[j][i] = a[i][j] = 0.0;
	}

	return t;
}

void symmetric_eigen(const Matrix m, int n, Vector vals, VecArray vecs)
{
	tridiag *t;
	Matrix M;
	Vector tmp;
	int i, j, k, s;

	M = dupMatrix(m);
	s = MatrixSize1(m);
	t = householder(M);
	tridiag_eigen(t, n, vals, vecs);
	
	if(vecs) 
	{
		tmp = newVector(s);
		for(i = 0; i < n; i++)
		{
			for(j = 0; j < s; j++)
			{
				tmp[j] = 0.0;
				for(k = 0; k < s; k++) 
					tmp[j] += vecs[i][k]*M[j][k];
			}
			for(j = 0; j < s; j++) vecs[i][j] = tmp[j];
		}
		deleteVector(tmp);
	}
	
	deletetridiag(t);
	deleteMatrix(M);
}

/* find w that maximizes w.A.w / w.B.w 
 *
 * A and B must be symmetric matrices.  B must be positive definite.
 */
Vector maximize_quadratic_ratio2(const Matrix A, const Matrix B, int niter,
	const Vector w0)
{
	Matrix L, Li, Lit, Lt, X0, X;
	Vector w, t;
	int N, i;

	g_assert(A);
	g_assert(B);
	
	g_assert(Matrixissquare(A));
	g_assert(Matrixissquare(B));
	N = MatrixSize1(A);
	g_assert(N == MatrixSize1(B));
	
	L = choleskydecompose(B);
	if(!L)
	{
		fprintf(stderr, "Quadratic ratio maximize: "
				"Cholesky decomposition failed\n");
		return 0;
	}
	Lt = transposeMatrix(L);
	Li = LMatrixinvert(L);
	deleteMatrix(L);
	Lit = transposeMatrix(Li);
	
	if(w0) t = MatrixVectormultiply(Lt, w0);
	else
	{
		t = newVector(N);
		fillVector(t, 1.0);
		
	}
	w = 0;
	
	X0 = Matrixmultiply(Li, A);
	deleteMatrix(Li);
	X = Matrixmultiply(X0, Lit);
	deleteMatrix(X0);

	for(i = 0; i < niter; i++)
	{
		w = MatrixVectormultiply(X, t);
		normalizeVector(w);
		deleteVector(t);
		t = w;
	}
	
	deleteMatrix(X);
	
	w = MatrixVectormultiply(Lit, t);
	deleteVector(t);

	normalizeVector(w);

	return w;
}

Vector maximize_quadratic_ratio(const Matrix A, const Matrix B)
{
	Matrix L, Li, Lit, X0, X;
	Vector w, vals;
	VecArray vecs;
	int N;

	g_assert(A);
	g_assert(B);
	
	g_assert(Matrixissquare(A));
	g_assert(Matrixissquare(B));
	N = MatrixSize1(A);
	g_assert(N == MatrixSize1(B));
	
	L = choleskydecompose(B);
	if(!L)
	{
		fprintf(stderr, "Quadratic ratio maximize: "
				"Cholesky decomposition failed\n");
		return 0;
	}
	Li = LMatrixinvert(L);
	deleteMatrix(L);
	Lit = transposeMatrix(Li);
	
	vals = newVector(1);
	vecs = newpopulatedVecArray(1, N);
	
	X0 = Matrixmultiply(Li, A);
	deleteMatrix(Li);
	X = Matrixmultiply(X0, Lit);
	deleteMatrix(X0);
	symmetric_eigen(X, 1, vals, vecs);
	deleteMatrix(X);
	
	w = MatrixVectormultiply(Lit, vecs[0]);

	deleteVector(vals);
	deleteVecArrayandVectors(vecs);
	
	normalizeVector(w);

	return w;
}

void quadratic_ratio_solve(const Matrix A, const Matrix B, Vector vals,
	VecArray vecs)
{
	Matrix L, Li, Lit, X0, X;
	Vector w;
	int i, N, q;

	g_assert(A);
	g_assert(B);
	
	g_assert(Matrixissquare(A));
	g_assert(Matrixissquare(B));
	N = MatrixSize1(A);
	g_assert(N == MatrixSize1(B));

	q = N;
	
	L = choleskydecompose(B);
	if(!L)
	{
		fprintf(stderr, "Quadratic ratio solve: "
				"Cholesky decomposition failed\n");
		return;
	}
	Li = LMatrixinvert(L);
	deleteMatrix(L);
	Lit = transposeMatrix(Li);
	
	X0 = Matrixmultiply(Li, A);
	deleteMatrix(Li);
	X = Matrixmultiply(X0, Lit);
	deleteMatrix(X0);
	printf("Li A Lit:");
	symmetric_eigen(X, q, vals, vecs);
	saveMatrixaspgm(X, "X.pgm");
	deleteMatrix(X);

	
	for(i = 0; i < q; i++)
	{
		w = MatrixVectormultiply(Lit, vecs[0]);
		normalizeVector(w);
		copytoVector(vecs[0], w);
		deleteVector(w);
	}
}
