#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>

#include "errorellipse.h"

void ErrorEllipse_calc_variances(ErrorEllipse ee);
void ErrorEllipse_calc_ellipse(ErrorEllipse ee);

void ErrorEllipse_calc_variances(ErrorEllipse ee)
{
	double c, s, det;
	double A, B, C;
	
	s = sin(ee->angle);
	c = cos(ee->angle);

	A = c*c/(ee->minor*ee->minor) + s*s/(ee->major*ee->major);
	B = s*s/(ee->minor*ee->minor) + c*c/(ee->major*ee->major);
	C = c*s*(1.0/(ee->major*ee->major) - 1.0/(ee->minor*ee->minor));

	det = A*B - C*C;

	ee->sigxx =  B/det;
	ee->sigyy =  A/det;
	ee->sigxy = -C/det;
}

void ErrorEllipse_calc_ellipse(ErrorEllipse ee)
{
	double det, A, B, C, s, c, t;

	det = ee->sigxx*ee->sigyy - ee->sigxy*ee->sigxy;
	A = ee->sigyy/det;
	B = ee->sigxx/det;
	C = -ee->sigxy/det;

	ee->angle = -0.5*atan2(2.0*C, A-B);
	s = sin(ee->angle);
	c = cos(ee->angle);
	ee->minor = 1.0/sqrt(A*c*c + B*s*s - 2.0*C*c*s);
	ee->major = 1.0/sqrt(A*s*s + B*c*c + 2.0*C*c*s);

	if(ee->minor > ee->major)
	{
		t = ee->major;
		ee->major = ee->minor;
		ee->minor = t;
		ee->angle += M_PI/2.0;
	}
	while(ee->angle < -M_PI/2.0) ee->angle += M_PI;
	while(ee->angle >= M_PI/2.0) ee->angle -= M_PI;
}

ErrorEllipse newErrorEllipse(double major, double minor, double angle)
{
	ErrorEllipse ee;

	ee = g_new(struct _ErrorEllipse, 1);

	ee->major = major;
	ee->minor = minor;
	ee->angle = angle;

	ErrorEllipse_calc_variances(ee);

	return ee;
}

ErrorEllipse newErrorEllipsefromvariances(
		double sigxx, double sigyy, double sigxy)
{
	ErrorEllipse ee;

	ee = g_new(struct _ErrorEllipse, 1);

	ee->sigxx = sigxx;
	ee->sigyy = sigyy;
	ee->sigxy = sigxy;

	ErrorEllipse_calc_ellipse(ee);

	return ee;
}

ErrorEllipse dupErrorEllipse(const ErrorEllipse ee)
{
	return newErrorEllipse(ee->major, ee->minor, ee->angle);
}

void deleteErrorEllipse(ErrorEllipse ee)
{
	g_free(ee);
}

void printErrorEllipse(ErrorEllipse ee)
{
	printf("major = %f minor = %f angle = %f\n", 
				ee->major, ee->minor, ee->angle);
	printf("sigxx = %f sigyy = %f sigxy = %f\n", 
				ee->sigxx, ee->sigyy, ee->sigxy);
}

/* Algorithm from paper by J. P. Wild, Australian Journal of Physics, 1970,
 * 23, pp113-115
 */
void convolveErrorEllipse(ErrorEllipse ee1, const ErrorEllipse ee2)
{
	double minussin, minuscos, plus, minus;

	plus  = (ee1->major)*(ee1->major)+(ee1->minor)*(ee1->minor);
	plus += (ee2->major)*(ee2->major)+(ee2->minor)*(ee2->minor);

	minussin  = ( (ee1->major)*(ee1->major) -
	              (ee1->minor)*(ee1->minor) ) * sin(2.0*ee1->angle);
	minussin += ( (ee2->major)*(ee2->major) -
	              (ee2->minor)*(ee2->minor) ) * sin(2.0*ee2->angle);

	minuscos  = ( (ee1->major)*(ee1->major) -
	              (ee1->minor)*(ee1->minor) ) * cos(2.0*ee1->angle);
	minuscos += ( (ee2->major)*(ee2->major) -
	              (ee2->minor)*(ee2->minor) ) * cos(2.0*ee2->angle);

	minus = sqrt(minussin*minussin + minuscos*minuscos);

	ee1->major = sqrt(0.5*(plus + minus));
	ee1->minor = sqrt(0.5*(plus - minus));
	ee1->angle = 0.5*atan2(minussin, minuscos);
	
	ErrorEllipse_calc_variances(ee1);
}

void deconvolveErrorEllipse(ErrorEllipse ee1, const ErrorEllipse ee2)
{
	double minussin, minuscos, plus, minus, temp;

	plus  = (ee1->major)*(ee1->major)+(ee1->minor)*(ee1->minor);
	plus -= (ee2->major)*(ee2->major)+(ee2->minor)*(ee2->minor);

	minussin  = ( (ee1->major)*(ee1->major) -
	              (ee1->minor)*(ee1->minor) ) * sin(2.0*ee1->angle);
	minussin -= ( (ee2->major)*(ee2->major) -
	              (ee2->minor)*(ee2->minor) ) * sin(2.0*ee2->angle);

	minuscos  = ( (ee1->major)*(ee1->major) -
	              (ee1->minor)*(ee1->minor) ) * cos(2.0*ee1->angle);
	minuscos -= ( (ee2->major)*(ee2->major) -
	              (ee2->minor)*(ee2->minor) ) * cos(2.0*ee2->angle);

	minus = sqrt(minussin*minussin + minuscos*minuscos);

	if(minus > plus)
	{
		temp = minus;
		minus = plus;
		plus = temp;
		minussin = -minussin;
		minuscos = minuscos;
		printf("Cannot deconv\n");
	}
	ee1->major = sqrt(0.5*(plus + minus));
	ee1->minor = sqrt(0.5*(plus - minus));
	ee1->angle = 0.5*atan2(minussin, minuscos);
	
	ErrorEllipse_calc_variances(ee1);
}

// elongates error bar in radial direction by convolving error ellipse with
// radial gaussian with width D*(BW/F)*radius, where D is a form factor,
// typically 0.05 (via Ed Fomalont)

void bwsmearErrorEllipse(ErrorEllipse ee, double D, double x, double y,
	double freq, double bw)
{
	ErrorEllipse bwsmear;
	double width, theta;

	width = D*sqrt(x*x+y*y)*bw/freq;
	theta = atan2(x, y);	
	
	bwsmear = newErrorEllipse(width, 0.0, theta);
	
	convolveErrorEllipse(ee, bwsmear);
}

void rotateErrorEllipse(ErrorEllipse ee, double angle)
{
	ee->angle += angle;
	ErrorEllipse_calc_variances(ee);
}

double ErrorEllipsearea(const ErrorEllipse ee)
{
	return 0.25*M_PI*ee->major*ee->minor;
}

void renderErrorEllipsetoMatrix(const ErrorEllipse ee, Matrix M, 
	double dx, double dy, double scale)
{
	double det, A, B, C, x, y;
	int i, j, m, n;

	n = MatrixSize1(M);
	m = MatrixSize2(M);

	det = ee->sigxx*ee->sigyy - ee->sigxy*ee->sigxy;
	A = ee->sigyy/det;
	B = ee->sigxx/det;
	C = -ee->sigxy/det;
	for(j = 0; j < n; j++) 
	{
		y = (j-dy)/scale;
		for(i = 0; i < m; i++)
		{
			x = (i-dx)/scale;
			M[j][i] = exp(-0.5*(A*x*x + B*y*y + 2.0*C*x*y));
		}
	}
}

void rendersolidErrorEllipsetoMatrix(const ErrorEllipse ee, Matrix M, 
	double dx, double dy, double scale)
{
	double det, A, B, C, x, y, twologtwo;
	double s, c, t;
	int i, j, m, n, X, Y;

	twologtwo = 2.0*log(2.0);

	n = MatrixSize1(M);
	m = MatrixSize2(M);

	if(ee->minor*scale < 1.5)
	{
		zeroMatrix(M);

		s = scale*ee->major*sin(ee->angle);
		c = scale*ee->major*cos(ee->angle);

		if(abs(s) > abs(c))
		{
			t = c/s;
			for(i = 0; i <= abs(s); i++)
			{
				X = dx+i+0.5;
				Y = dy+i*t+0.5;
				M[Y % n][X % m] = 1.0;
				X = dx-i+0.5;
				Y = dy-i*t+0.5;
				M[Y % n][X % m] = 1.0;
			}
		}
		else
		{
			t = s/c;
			for(j = 0; j <= abs(c); j++)
			{
				X = dx+j*t+0.5;
				Y = dy+j+0.5;
				M[Y % n][X % m] = 1.0;
				X = dx-j*t+0.5;
				Y = dy-j+0.5;
				M[Y % n][X % m] = 1.0;
			}
		}
	}
	else
	{
		det = ee->sigxx*ee->sigyy - ee->sigxy*ee->sigxy;
		A = ee->sigyy/det;
		B = ee->sigxx/det;
		C = -ee->sigxy/det;
		for(j = 0; j < n; j++) 
		{
			y = (j-dy)/scale;
			for(i = 0; i < m; i++)
			{
				x = (i-dx)/scale;
				if(A*x*x + B*y*y + 2.0*C*x*y < twologtwo)
					M[j][i] = 1.0;
				else M[j][i] = 0.0;
			}
		}
	}
}

