#include <config.h>

#include <math.h>
#include <string.h>
#include <stdio.h>
#include <glib.h>
#include "cube.h"

Cube newstridedCube(int n1, int n2, int n3, int stride2, int stride3)
{
	Cube C;
	CubeTypePointer data;
	int i1, i2;

	g_assert(n2 <= stride2);
	g_assert(n3 <= stride3);

	data = g_new(CubeType, n1*stride2*stride3+CUBEVALUESPERBLOCK);
	data += CUBEVALUESPERBLOCK;
	C = g_new(CubeTypePointer*, ((n1>1) ? (n1 + CUBESTUBINDICES) :
					      (2  + CUBESTUBINDICES) ));
	C += MATRIXSTUBINDICES;

	CubeStub(C)->datastart = data;
	CubeStub(C)->n1 = n1;
	CubeStub(C)->n2 = n2;
	CubeStub(C)->n3 = n3;
	CubeStub(C)->stride2 = stride2;
	CubeStub(C)->stride3 = stride3;
	Cuberefcount(C) = 1;
	
	for(i1 = 0; i1 < n1; i1++)
	{
		C[i1] = g_new(CubeTypePointer, stride2);
		for(i2 = 0; i2 < stride2; i2++)
			C[i1][i2] = data + i1*stride2*stride3 + i2*stride3;
	}
	if(n1 == 1) C[1] = C[0] + stride2*stride3;
	
	return C;
}

Cube refsubCube(const Cube C, int a1, int a2, int a3, int b1, int b2, int b3)
{
	Cube D;
	CubeTypePointer data;
	int n1, n2, n3;
	int i1, i2;
	int stride2, stride3;
	
	g_assert(C);
	
	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);

	if(a1 < 0) a1 = 0;
	if(a2 < 0) a2 = 0;
	if(a3 < 0) a3 = 0;
	if(b1 < 0) b1 = n1-1;
	if(b2 < 0) b2 = n2-1;
	if(b3 < 0) b3 = n3-1;
	
	if(b1 < a1 || b2 < a2 || b3 < a3 || b1 >= n1 || b2 >= n2 || b3 >= n3)
	{
		fprintf(stderr, "subCube: limit error\n");
		return 0;
	}

	D = g_new(CubeTypePointer*, ((n1>1) ? (n1 + CUBESTUBINDICES) :
	 				      (2  + CUBESTUBINDICES) ));
	D += MATRIXSTUBINDICES;
	CubeStub(D)->datastart = data = CubeStub(C)->datastart;
	CubeStub(D)->n1 = b1-a1+1;
	CubeStub(D)->n2 = b2-a2+1;
	CubeStub(D)->n3 = b3-a3+1;
	stride2 = CubeStub(D)->stride2 = CubeStub(C)->stride2;
	stride3 = CubeStub(D)->stride3 = CubeStub(C)->stride3;
	Cuberefcount(C)++;
	for(i1 = 0; i1 <= b1-a1; i1++)
	{
		D[i1] = g_new(CubeTypePointer, stride2);
		for(i2 = 0; i2 < stride2; i2++)
			D[i1][i2] = data + i1*stride2*stride3 + i2*stride3;
	}
	if(a1 == b1) D[1] = D[0] + stride2*stride3;
	
	return D;
}

void deleteCube(Cube C)
{
	int i1, n1;
	CubeStubType *cs;

	g_assert(C);

	n1 = CubeSize1(C);
	
	Cuberefcount(C)--;
	cs = CubeStub(C);
	if(Cuberefcount(C) <= 0)
	{
		for(i1 = 0; i1 < n1; i1++) g_free(C[i1]);
		if(cs->datastart-CUBEVALUESPERBLOCK == 0)
			fprintf(stderr, "ACK!!! cube data gone!\n");
		else
			g_free(cs->datastart-CUBEVALUESPERBLOCK);
	}
	
	g_free(cs);
}

void zeroCube(Cube C)
{
	int i1, i2, i3;
	int n1, n2, n3;
	CubeTypePointer *c1, c2;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);
	for(i1 = 0; i1 < n1; i1++)
	{
		c1 = C[i1];
		for(i2 = 0; i2 < n2; i2++)
		{
			c2 = c1[i2];
			for(i3 = 0; i3 < n3; i3++) c2[i3] = 0.0;
		}
	}
}

Cube dupCube(const Cube C)
{
	Cube D;
	int i1, i2, i3;
	int n1, n2, n3;
	CubeTypePointer *c1, c2;
	CubeTypePointer *d1, d2;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);

	D = newCube(n1, n2, n3);
	
	for(i1 = 0; i1 < n1; i1++)
	{
		c1 = C[i1];
		d1 = D[i1];
		for(i2 = 0; i2 < n2; i2++)
		{
			c2 = c1[i2];
			d2 = d1[i2];
			for(i3 = 0; i3 < n3; i3++) d2[i3] = c2[i3];
		}
	}
	
	return D;
}


Matrix Cubeslice1(const Cube C, int s1)
{
	Matrix M;
	int i2, i3;
	int n1, n2, n3;
	CubeTypePointer *c1, c2;
	MatrixTypePointer m1;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);
	
	if(s1 >= n1 || s1 < 0) return 0;

	c1 = C[s1];

	M = newMatrix(n2, n3);

	for(i2 = 0; i2 < n2; i2++)
	{
		c2 = c1[i2];
		m1 = M[i2];
		for(i3 = 0; i3 < n3; i3++) m1[i3] = c2[i3];
	}

	return M;
}

Matrix Cubeslice2(const Cube C, int s2)
{
	Matrix M;
	int i1, i3;
	int n1, n2, n3;
	CubeTypePointer c2;
	MatrixTypePointer m1;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);
	
	if(s2 >= n2 || s2 < 0) return 0;

	M = newMatrix(n1, n3);

	for(i1 = 0; i1 < n1; i1++)
	{
		c2 = C[i1][s2];
		m1 = M[i1];
		for(i3 = 0; i3 < n3; i3++) m1[i3] = c2[i3];
	}

	return M;
}

Matrix Cubeslice3(const Cube C, int s3)
{
	Matrix M;
	int i1, i2;
	int n1, n2, n3;
	CubeTypePointer *c1;
	MatrixTypePointer m1;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);
	
	if(s3 >= n3 || s3 < 0) return 0;

	M = newMatrix(n1, n2);

	for(i1 = 0; i1 < n1; i1++)
	{
		c1 = C[i1];
		m1 = M[i1];
		for(i2 = 0; i2 < n2; i2++) m1[i2] = c1[i2][s3];
	}
	
	return M;
}

CubeType Cubemaxvalue(const Cube C, int *a, int *b, int *c)
{
	int i1, i2, i3;
	int n1, n2, n3;
	int m1, m2, m3;
	CubeType max;
	CubeTypePointer *c1, c2;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);
	max = C[0][0][0];
	m1 = m2 = m3 = 0l;
	for(i1 = 0; i1 < n1; i1++)
	{
		c1 = C[i1];
		for(i2 = 0; i2 < n2; i2++)
		{
			c2 = c1[i2];
			for(i3 = 0; i3 < n3; i3++) if(c2[i3] > max)
			{
				m1 = i1;
				m2 = i2;
				m3 = i3;
				max = c2[i3];
			}
		}
	}
	if(a) *a = m1;
	if(b) *b = m2;
	if(c) *c = m3;
	return max;
}

CubeType Cubeminvalue(const Cube C, int *a, int *b, int *c)
{
	int i1, i2, i3;
	int n1, n2, n3;
	int m1, m2, m3;
	CubeType min;
	CubeTypePointer *c1, c2;

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);
	min = C[0][0][0];
	m1 = m2 = m3 = 0l;
	for(i1 = 0; i1 < n1; i1++)
	{
		c1 = C[i1];
		for(i2 = 0; i2 < n2; i2++)
		{
			c2 = c1[i2];
			for(i3 = 0; i3 < n3; i3++) if(c2[i3] < min)
			{
				m1 = i1;
				m2 = i2;
				m3 = i3;
				min = c2[i3];
			}
		}
	}
	if(a) *a = m1;
	if(b) *b = m2;
	if(c) *c = m3;
	return min;
}

/* returns covariance matrix */
Matrix Cubepeakup(const Cube C, int x1, int x2, int x3, int p1, int p2, int p3, 
	double *y1, double *y2, double *y3, double *peak)
{
	int q1, q2, q3;
	int i1, i2, i3;
	int n1, n2, n3;
	int u, v, res;
	Matrix M, M2;
	Vector V, V2;
	double f[10];

	g_assert(C);

	n1 = CubeSize1(C);
	n2 = CubeSize2(C);
	n3 = CubeSize3(C);

	V = newVector(10);
	M = newMatrix(10, 10);
	zeroVector(V);
	zeroMatrix(M);

	f[9] = 1.0;

	for(q1 = -p1; q1 <= p1; q1++)
	{
		i1 = x1 + q1;
		if(i1 < 0) 
		{
			if(p1 > 1) continue;
			else i1 += n1;
		}
		else if(i1 >= n1) 
		{
			if(p1 > 1) continue;
			else i1 -= n1;
		}
		f[0] = q1*q1;
		f[6] = q1;
		for(q2 = -p2; q2 <= p2; q2++)
		{
			i2 = x2 + q2;
			if(i2 < 0) 
			{
				if(p2 > 1) continue;
				else i2 += n2;
			}
			else if(i2 >= n2) 
			{
				if(p2 > 1) continue;
				else i2 -= n2;
			}
			f[1] = q2*q2;
			f[3] = q1*q2;
			f[7] = q2;
			for(q3 = -p3; q3 <= p3; q3++)
			{
				i3 = x3 + q3;
				if(i3 < 0) 
				{
					if(p3 > 1) continue;
					else i3 += n3;
				}
				else if(i3 >= n3) 
				{
					if(p3 > 1) continue;
					else i3 -= n3;
				}
				f[2] = q3*q3;
				f[4] = q1*q3;
				f[5] = q2*q3;
				f[8] = q3;
				for(u = 0; u < 10; u++) 
					V[u] += f[u]*C[i1][i2][i3];
				for(u = 0; u < 10; u++) for(v = 0; v< 10; v++)
					M[u][v] += f[u]*f[v];
			}
		}
	}

	res = gaussjordan(M, V);
	g_assert(res == 1);

	deleteMatrix(M);

	M = newMatrix(3, 3);
	V2 = newVector(3);

	M[0][0] = 2.0*V[0];
	M[1][1] = 2.0*V[1];
	M[2][2] = 2.0*V[2];
	M[0][1] = M[1][0] = V[3];
	M[0][2] = M[2][0] = V[4];
	M[1][2] = M[2][1] = V[5];
	V2[0] = -V[6];
	V2[1] = -V[7];
	V2[2] = -V[8];

	if(peak) *peak = V[9];

	M2 = dupMatrix(M);

	deleteVector(V);
	
	res = gaussjordan(M, V2);
	deleteMatrix(M);
	
	if(res != 1)
	{
		fprintf(stderr, "Cubepeakup : no solution\n");
		if(y1) *y1 = x1;
		if(y2) *y2 = x2;
		if(y3) *y3 = x3;
		deleteVector(V2);
		deleteMatrix(M2);
		return 0;
	}
	
	if(y1) *y1 = x1 + V2[0];
	if(y2) *y2 = x2 + V2[1];
	if(y3) *y3 = x3 + V2[2];

	deleteVector(V2);

	return M2;
}
