/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <glib.h>
#include "plugin.h"
#include "rgbdraw.h"

void drawplanetarynebula(struct viewer *v, float *diammin, int x, int y, int r, int g, int b)
{
	float radius;

	radius = *diammin*v->scale/10800.0;
	if(radius < 1.5) radius = 1.5;

	drawcircle(v->image, x, y, radius, r, g, b);
	drawvline(v->image, x, y-ceil(2.0*radius), y-ceil(radius), r, g, b);
	drawhline(v->image, x-ceil(2.0*radius), y, x-ceil(radius), r, g, b);
	drawvline(v->image, x, y+ceil(radius), y+ceil(2.0*radius), r, g, b);
	drawhline(v->image, x+ceil(radius), y, x+ceil(2.0*radius), r, g, b);
}

void printplanetarynebula(struct viewer *v, float *data, double scale, 
	FILE *out)
{
	float radius;

	radius = *data*scale/10800.0;
	if(radius < 1.5) radius = 1.5;
	fprintf(out, "2 copy %6.2f 0 360 arc stroke\n", radius);
	fprintf(out, "moveto 0 %6.2f rmoveto 0 %6.2f rlineto\n",
		radius, radius);
	fprintf(out, "0 %6.2f rmoveto 0 %6.2f rlineto\n", -4.0*radius, radius);
	fprintf(out, "%6.2f %6.2f rmoveto %6.2f 0 rlineto\n",
		-2.0*radius, radius, radius);
	fprintf(out, "%6.2f 0 rmoveto %6.2f 0 rlineto stroke\n",
		2.0*radius, radius);
}

void drawglobularcluster(struct viewer *v, float *diammin, int x, int y, int r, int g, int b)
{
	float radius;

	radius = *diammin*v->scale/10800.0;
	if(radius < 2.8) radius = 2.8;

	drawcircle(v->image, x, y, radius, r, g, b);
	drawvline(v->image, x, y-floor(radius), y+floor(radius), r, g, b);
	drawhline(v->image, x-floor(radius), y, x+floor(radius), r, g, b);
}

void printglobularcluster(struct viewer *v, float *data, double scale, 
	FILE *out)
{
	float radius;

	radius = *data*scale/10800.0;
	if(radius < 2.0) radius = 2.0;
	fprintf(out, "2 copy %6.2f 0 360 arc stroke\n", radius);
	fprintf(out, "moveto 0 %6.2f rmoveto 0 %6.2f rlineto\n",
		radius, -2.0*radius);
	fprintf(out, "%6.2f %6.2f rmoveto\n", radius, radius);
	fprintf(out, "%6.2f 0 rlineto stroke\n", -2.0*radius);
}

void drawopencluster(struct viewer *v, float *diammin, int x, int y, int r, int g, int b)
{
	float radius;

	radius = *diammin*v->scale/10800.0;
	if(radius < 2.8) radius = 2.8;

	drawcircle(v->image, x, y, radius, r, g, b);
}

void drawcircle7(struct viewer *v, float *d, int x, int y, int r, int g, int b)
{
	drawcircle(v->image, x, y, 7.0, r, g, b);
	drawcircle(v->image, x, y, 7.9, r, g, b);
}

void printopencluster(struct viewer *v, float *data, double scale, FILE *out)
{
	float radius;

	radius = *data*scale/10800.0;
	if(radius < 3.0) radius = 3.0;
	fprintf(out, "[1 2] 0 setdash %6.2f 0 360 arc stroke [] 0 setdash\n",
		radius);
}

void drawpulsar(struct viewer *v, float *extra, int x, int y, int r, int g, int b)
{
	fillcircle(v->image, x, y, 1.8, r, g, b);
}

void printpulsar(struct viewer *v, float *data, double scale, FILE *out)
{
	fprintf(out, "2 copy 3 0 360 arc stroke moveto 2 5 rmoveto -4 -10 rlineto stroke\n");
}

void drawgalaxy(struct viewer *v, float *ellipse, int x, int y, int r, int g, int b)
{
	float maj, min, pa;

	maj = ellipse[0]*v->scale/10800.0;
	min = ellipse[1]*v->scale/10800.0;
	pa = ellipse[2];

	if(maj < 2) 
	{
		drawhline(v->image, x, y, x+1, r, g, b);
		return;
	}
	if(min < 1) min = 1;
	if(maj > 6 && min < 2) min = 2;
	if(maj > 12 && min < 3) min = 3;

	if(x+maj+1 >= gdk_pixbuf_get_width(v->image)) return;
	if(x-maj-1 < 0) return;
	if(y+maj+1 >= gdk_pixbuf_get_height(v->image)) return;
	if(y-maj-1 < 0) return;

	drawellipse(v->image, x, y, maj, min, -pa, r, g, b);
}

void printgalaxy(struct viewer *v, float *data, double scale, FILE *out)
{
	float maj, min, pa;

	maj = data[0]*v->scale/10800.0;
	min = data[1]*v->scale/10800.0;
	pa = data[2]*180/PI;

	if(maj < 1.5) maj = 1.5;
	if(min < 1) min = 1;

	fprintf(out, "gsave translate\n");
	fprintf(out, "%6.2f rotate\n", pa);
	fprintf(out, "%6.2f 1 scale\n", min/maj);
	fprintf(out, "0 0 %6.2f 0 360 arc stroke\n", maj);
	fprintf(out, "grestore\n");
}

void printcircle7(struct viewer *v, float *data, double scale, FILE *out)
{
}

GList *pluginlist = 0;

void linkplugin(const char *name, plugindrawfunc drawfunc, 
		pluginprintfunc printfunc)
{
	plugindata *pd;

	pd = (plugindata *)malloc(sizeof(plugindata));
	pd->name = g_strdup(name);
	pd->drawfunc = drawfunc;
	pd->printfunc = printfunc;
	pluginlist = g_list_append(pluginlist, pd);
}

plugindata *getplugindata(const char *name)
{
	GList *l;

	for(l = pluginlist; l != 0; l = l-> next)
		if(strcmp(((plugindata *)(l->data))->name, name) == 0)
			return (plugindata *)(l->data);
	printf("Warning -- cannot link to plugin%s\n", name);
	return 0;
}

void initplugins()
{
	linkplugin("DrawPlanetaryNebula", drawplanetarynebula, 
					  printplanetarynebula);
	linkplugin("DrawGlobularCluster", drawglobularcluster, 
					  printglobularcluster);
	linkplugin("DrawOpenCluster",     drawopencluster, 
					  printopencluster);
	linkplugin("DrawGalaxy",          drawgalaxy, 
					  printgalaxy);
	linkplugin("DrawPulsar", drawpulsar, printpulsar);
	linkplugin("DrawCircle", drawcircle7, printcircle7);
}
