/***************************************************************************
 *   Copyright (C) 2007 by Walter Brisken                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
//===========================================================================
// SVN properties (DO NOT CHANGE)
//
// $Id: mark5_format_mark5b.c 1309 2009-07-13 20:42:40Z WalterBrisken $
// $HeadURL: https://svn.atnf.csiro.au/difx/libraries/mark5access/branches/difx-1.5/mark5access/mark5_format_mark5b.c $
// $LastChangedRevision: 1309 $
// $Author: WalterBrisken $
// $LastChangedDate: 2009-07-13 14:42:40 -0600 (Mon, 13 Jul 2009) $
//
//============================================================================

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mark5access/mark5_stream.h"

#define MK5B_PAYLOADSIZE 10000

const uint32_t mark5bSync       = 0xABADDEED;
const uint32_t mark5cCompatSync = 0xF00FF00F;	/* FIXME: replace with actual */

/* the high mag value for 2-bit reconstruction */
static const float HiMag = 3.3359;

struct mark5_format_mark5b
{
	int nbitstream;
	int kday;	/* kilo-mjd -- ie 51000, 52000, ... */
};

static float lut1bit[256][8];
static float lut2bit[256][4];
static float zeros[8];

static void initluts()
{
	int b, i, s, m, l;
	const float lut2level[2] = {1.0, -1.0};
	const float lut4level[4] = {-HiMag, 1.0, -1.0, HiMag};

	for(i = 0; i < 8; i++)
	{
		zeros[i] = 0.0;
	}

	for(b = 0; b < 256; b++)
	{
		/* lut1bit */
		for(i = 0; i < 8; i++)
		{
			l = (b>>i)&1;
			lut1bit[b][i] =  lut2level[l];
		}

		/* lut2bit */
		for(i = 0; i < 4; i++)
		{
			s = i*2;	/* 0, 2, 4, 6 */
			m = s+1;	/* 1, 3, 5, 7 */
			l = ((b>>s)&1) + (((b>>m)&1)<<1);
			lut2bit[b][i] =  lut4level[l];
		}
	}
}

static int findfirstframe(const uint8_t *data, int bytes, uint32_t syncword)
{
	int i;
	uint8_t sb0, sb1, sb2, sb3;

	sb0 = (syncword >>  0) & 0xFF;
	sb1 = (syncword >>  8) & 0xFF;
	sb2 = (syncword >> 16) & 0xFF;
	sb3 = (syncword >> 24) & 0xFF;

	bytes -= 10020; 
	if(bytes < 0)	/* enough bytes to decode? */
	{
		return -1;
	}
	
	/* look for two consecutive frame sync words */
	for(i = 0; i < bytes; i++)
	{
		if(data[0]     == sb0 &&
		   data[1]     == sb1 &&
		   data[2]     == sb2 &&
		   data[3]     == sb3 &&
		   data[10016] == sb0 &&
		   data[10017] == sb1 &&
		   data[10018] == sb2 &&
		   data[10019] == sb3    )
		{
			return i;
		}
		data++;
	}

	return -1;
}

static int mark5_stream_frame_num_mark5b(const struct mark5_stream *ms)
{
	return ms->frame[4] + (ms->frame[5] & 0x7F)*256;
}

static int mark5_stream_frame_time_mark5b(const struct mark5_stream *ms,
	int *mjd, int *sec, double *ns)
{
	struct mark5_format_mark5b *m;
	const uint8_t *buf;
	int i, framenum;
	uint8_t nibs[16];

	m = (struct mark5_format_mark5b *)(ms->formatdata);

	buf = ms->frame + 8;

	framenum = mark5_stream_frame_num_mark5b(ms);

	for(i = 0; i < 4; i++)
	{
		nibs[2*i+0] = buf[3-i] >> 4;
		nibs[2*i+1] = buf[3-i] & 0x0F;
		nibs[2*i+8] = buf[7-i] >> 4;
		nibs[2*i+9] = buf[7-i] & 0x0F;
	}

	if(mjd)
	{
		*mjd = m->kday + nibs[0]*100 + nibs[1]*10 + nibs[2];
	}
	if(sec) 
	{
		*sec = nibs[3]*10000 + nibs[4]*1000 + nibs[5]*100 + nibs[6]*10
	        	+ nibs[7];
	}
	if(ns)
	{
		if(ms->framens > 0)
		{
			*ns = ms->framens*framenum;
		}
		else
		{
			*ns = nibs[8]*100000000 + nibs[9]*10000000 
				+ nibs[10]*1000000+ nibs[11]*100000;
			/* "unround" the number */
			*ns = 156250*(((int)(*ns)+156249)/156250);
		}
	}

	return 0;
}

static int mark5_stream_frame_time_mark5cb(const struct mark5_stream *ms,
	int *mjd, int *sec, double *ns)
{
	struct mark5_format_mark5b *m;
	const uint8_t *buf;
	int n;	/* frame number within the 1 second period */
	int i;
	uint8_t nibs[8];

	m = (struct mark5_format_mark5b *)(ms->formatdata);

	buf = ms->frame + 8;
	
	/* 15 lowest bits of second 32-bit word */
	n = mark5_stream_frame_num_mark5b(ms);

	buf = ms->frame + 8;

	for(i = 0; i < 4; i++)
	{
		nibs[2*i+0] = buf[3-i] >> 4;
		nibs[2*i+1] = buf[3-i] & 0x0F;
	}

	if(mjd)
	{
		*mjd = m->kday + nibs[0]*100 + nibs[1]*10 + nibs[2];
	}
	if(sec) 
	{
		*sec = nibs[3]*10000 + nibs[4]*1000 + nibs[5]*100 + nibs[6]*10
	        	+ nibs[7];
	}
	if(ns)
	{
		*ns = ms->framens*n;
	}

	return 0;
}

static int mark5_format_mark5b_fixmjd(struct mark5_stream *ms, int refmjd)
{
	struct mark5_format_mark5b *m;
	int n;

	if(!ms)
	{
		return -1;
	}
	
	m = (struct mark5_format_mark5b *)(ms->formatdata);
	if(m->kday == 0)
	{
		n = (refmjd - ms->mjd + 500) / 1000;
		ms->mjd += n*1000;
		m->kday = n*1000;

		return 1;
	}

	return 0;
}

/************************* decode routines **************************/

static int mark5b_decode_1bitstream_1bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		o++;
		data[0][o] = fp[1];
		o++;
		data[0][o] = fp[2];
		o++;
		data[0][o] = fp[3];
		o++;
		data[0][o] = fp[4];
		o++;
		data[0][o] = fp[5];
		o++;
		data[0][o] = fp[6];
		o++;
		data[0][o] = fp[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp-8*nblank;
}

static int mark5b_decode_1bitstream_1bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		o++;
		data[0][o] = fp[2];
		o++;
		data[0][o] = fp[4];
		o++;
		data[0][o] = fp[6];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp-8*nblank;
}

static int mark5b_decode_1bitstream_1bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		o++;
		data[0][o] = fp[4];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp-8*nblank;
}

static int mark5b_decode_1bitstream_1bit_decimation8(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation/8;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp-8*nblank;
}

static int mark5b_decode_2bitstream_1bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		o++;
		data[0][o] = fp[2];
		data[1][o] = fp[3];
		o++;
		data[0][o] = fp[4];
		data[1][o] = fp[5];
		o++;
		data[0][o] = fp[6];
		data[1][o] = fp[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 4*nblank;
}

static int mark5b_decode_2bitstream_1bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		o++;
		data[0][o] = fp[4];
		data[1][o] = fp[5];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 4*nblank;
}

static int mark5b_decode_2bitstream_1bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation/4;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];
		data[1][o] = fp[1];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 4*nblank;
}

static int mark5b_decode_4bitstream_1bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];
		o++;
		data[0][o] = fp[4];
		data[1][o] = fp[5];
		data[2][o] = fp[6];
		data[3][o] = fp[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 2*nblank;
}

static int mark5b_decode_4bitstream_1bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 2*nblank;
}

static int mark5b_decode_4bitstream_1bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation/2;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 2*nblank;
}

static int mark5b_decode_8bitstream_1bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];
		data[4][o] = fp[4];
		data[5][o] = fp[5];
		data[6][o] = fp[6];
		data[7][o] = fp[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_8bitstream_1bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i += 2;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];
		data[4][o] = fp[4];
		data[5][o] = fp[5];
		data[6][o] = fp[6];
		data[7][o] = fp[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_8bitstream_1bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut1bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];
		data[4][o] = fp[4];
		data[5][o] = fp[5];
		data[6][o] = fp[6];
		data[7][o] = fp[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_16bitstream_1bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = zeros;
			nblank++;
			i += 2;
		}
		else
		{
			fp0 = lut1bit[buf[i]];
			i++;
			fp1 = lut1bit[buf[i]];
			i++;
		}

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp0[4];
		data[5][o]  = fp0[5];
		data[6][o]  = fp0[6];
		data[7][o]  = fp0[7];
		data[8][o]  = fp1[0];
		data[9][o]  = fp1[1];
		data[10][o] = fp1[2];
		data[11][o] = fp1[3];
		data[12][o] = fp1[4];
		data[13][o] = fp1[5];
		data[14][o] = fp1[6];
		data[15][o] = fp1[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_16bitstream_1bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = zeros;
			nblank++;
			i += 4;
		}
		else
		{
			fp0 = lut1bit[buf[i]];
			i++;
			fp1 = lut1bit[buf[i]];
			i += 3;
		}

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp0[4];
		data[5][o]  = fp0[5];
		data[6][o]  = fp0[6];
		data[7][o]  = fp0[7];
		data[8][o]  = fp1[0];
		data[9][o]  = fp1[1];
		data[10][o] = fp1[2];
		data[11][o] = fp1[3];
		data[12][o] = fp1[4];
		data[13][o] = fp1[5];
		data[14][o] = fp1[6];
		data[15][o] = fp1[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_16bitstream_1bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation*2 - 1;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = zeros;
			i++;
			fp1 = zeros;
			nblank++;
		}
		else
		{
			fp0 = lut1bit[buf[i]];
			i++;
			fp1 = lut1bit[buf[i]];
		}
		i += df;

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp0[4];
		data[5][o]  = fp0[5];
		data[6][o]  = fp0[6];
		data[7][o]  = fp0[7];
		data[8][o]  = fp1[0];
		data[9][o]  = fp1[1];
		data[10][o] = fp1[2];
		data[11][o] = fp1[3];
		data[12][o] = fp1[4];
		data[13][o] = fp1[5];
		data[14][o] = fp1[6];
		data[15][o] = fp1[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_32bitstream_1bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1, *fp2, *fp3;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = fp2 = fp3 = zeros;
			nblank++;
			i += 4;
		}
		else
		{
			fp0 = lut1bit[buf[i]];
			i++;
			fp1 = lut1bit[buf[i]];
			i++;
			fp2 = lut1bit[buf[i]];
			i++;
			fp3 = lut1bit[buf[i]];
			i++;
		}

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp0[4];
		data[5][o]  = fp0[5];
		data[6][o]  = fp0[6];
		data[7][o]  = fp0[7];
		data[8][o]  = fp1[0];
		data[9][o]  = fp1[1];
		data[10][o] = fp1[2];
		data[11][o] = fp1[3];
		data[12][o] = fp1[4];
		data[13][o] = fp1[5];
		data[14][o] = fp1[6];
		data[15][o] = fp1[7];
		data[16][o] = fp2[0];
		data[17][o] = fp2[1];
		data[18][o] = fp2[2];
		data[19][o] = fp2[3];
		data[20][o] = fp2[4];
		data[21][o] = fp2[5];
		data[22][o] = fp2[6];
		data[23][o] = fp2[7];
		data[24][o] = fp3[0];
		data[25][o] = fp3[1];
		data[26][o] = fp3[2];
		data[27][o] = fp3[3];
		data[28][o] = fp3[4];
		data[29][o] = fp3[5];
		data[30][o] = fp3[6];
		data[31][o] = fp3[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_32bitstream_1bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1, *fp2, *fp3;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = fp2 = fp3 = zeros;
			nblank++;
			i += 8;
		}
		else
		{
			fp0 = lut1bit[buf[i]];
			i++;
			fp1 = lut1bit[buf[i]];
			i++;
			fp2 = lut1bit[buf[i]];
			i++;
			fp3 = lut1bit[buf[i]];
			i += 5;
		}

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp0[4];
		data[5][o]  = fp0[5];
		data[6][o]  = fp0[6];
		data[7][o]  = fp0[7];
		data[8][o]  = fp1[0];
		data[9][o]  = fp1[1];
		data[10][o] = fp1[2];
		data[11][o] = fp1[3];
		data[12][o] = fp1[4];
		data[13][o] = fp1[5];
		data[14][o] = fp1[6];
		data[15][o] = fp1[7];
		data[16][o] = fp2[0];
		data[17][o] = fp2[1];
		data[18][o] = fp2[2];
		data[19][o] = fp2[3];
		data[20][o] = fp2[4];
		data[21][o] = fp2[5];
		data[22][o] = fp2[6];
		data[23][o] = fp2[7];
		data[24][o] = fp3[0];
		data[25][o] = fp3[1];
		data[26][o] = fp3[2];
		data[27][o] = fp3[3];
		data[28][o] = fp3[4];
		data[29][o] = fp3[5];
		data[30][o] = fp3[6];
		data[31][o] = fp3[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_32bitstream_1bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1, *fp2, *fp3;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation*4 - 3;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = zeros;
			i++;
			fp1 = zeros;
			i++;
			fp2 = zeros;
			i++;
			fp3 = zeros;
			nblank++;
		}
		else
		{
			fp0 = lut1bit[buf[i]];
			i++;
			fp1 = lut1bit[buf[i]];
			i++;
			fp2 = lut1bit[buf[i]];
			i++;
			fp3 = lut1bit[buf[i]];
		}
		i += df;

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp0[4];
		data[5][o]  = fp0[5];
		data[6][o]  = fp0[6];
		data[7][o]  = fp0[7];
		data[8][o]  = fp1[0];
		data[9][o]  = fp1[1];
		data[10][o] = fp1[2];
		data[11][o] = fp1[3];
		data[12][o] = fp1[4];
		data[13][o] = fp1[5];
		data[14][o] = fp1[6];
		data[15][o] = fp1[7];
		data[16][o] = fp2[0];
		data[17][o] = fp2[1];
		data[18][o] = fp2[2];
		data[19][o] = fp2[3];
		data[20][o] = fp2[4];
		data[21][o] = fp2[5];
		data[22][o] = fp2[6];
		data[23][o] = fp2[7];
		data[24][o] = fp3[0];
		data[25][o] = fp3[1];
		data[26][o] = fp3[2];
		data[27][o] = fp3[3];
		data[28][o] = fp3[4];
		data[29][o] = fp3[5];
		data[30][o] = fp3[6];
		data[31][o] = fp3[7];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

/************************ 2-bit decoders *********************/

static int mark5b_decode_2bitstream_2bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		o++;
		data[0][o] = fp[1];
		o++;
		data[0][o] = fp[2];
		o++;
		data[0][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 4*nblank;
}

static int mark5b_decode_2bitstream_2bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		o++;
		data[0][o] = fp[2];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 4*nblank;
}

static int mark5b_decode_2bitstream_2bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation/4;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 4*nblank;
}

static int mark5b_decode_4bitstream_2bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		o++;
		data[0][o] = fp[2];
		data[1][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 2*nblank;
}

static int mark5b_decode_4bitstream_2bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 2*nblank;
}

static int mark5b_decode_4bitstream_2bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation/2;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];
		data[1][o] = fp[1];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - 2*nblank;
}

static int mark5b_decode_8bitstream_2bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i++;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_8bitstream_2bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i += 2;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_8bitstream_2bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp = zeros;
			nblank++;
		}
		else
		{
			fp = lut2bit[buf[i]];
		}
		i += df;

		data[0][o] = fp[0];
		data[1][o] = fp[1];
		data[2][o] = fp[2];
		data[3][o] = fp[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_16bitstream_2bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = zeros;
			nblank++;
			i += 2;
		}
		else
		{
			fp0 = lut2bit[buf[i]];
			i++;
			fp1 = lut2bit[buf[i]];
			i++;
		}
		data[0][o] = fp0[0];
		data[1][o] = fp0[1];
		data[2][o] = fp0[2];
		data[3][o] = fp0[3];
		data[4][o] = fp1[0];
		data[5][o] = fp1[1];
		data[6][o] = fp1[2];
		data[7][o] = fp1[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_16bitstream_2bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = zeros;
			nblank++;
			i += 4;
		}
		else
		{
			fp0 = lut2bit[buf[i]];
			i++;
			fp1 = lut2bit[buf[i]];
			i += 3;
		}
		data[0][o] = fp0[0];
		data[1][o] = fp0[1];
		data[2][o] = fp0[2];
		data[3][o] = fp0[3];
		data[4][o] = fp1[0];
		data[5][o] = fp1[1];
		data[6][o] = fp1[2];
		data[7][o] = fp1[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_16bitstream_2bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation*2 - 1;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = zeros;
			i++;
			fp1 = zeros;
			nblank++;
		}
		else
		{
			fp0 = lut2bit[buf[i]];
			i++;
			fp1 = lut2bit[buf[i]];
		}
		i += df;
		data[0][o] = fp0[0];
		data[1][o] = fp0[1];
		data[2][o] = fp0[2];
		data[3][o] = fp0[3];
		data[4][o] = fp1[0];
		data[5][o] = fp1[1];
		data[6][o] = fp1[2];
		data[7][o] = fp1[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_32bitstream_2bit_decimation1(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1, *fp2, *fp3;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = fp2 = fp3 = zeros;
			nblank++;
			i += 4;
		}
		else
		{
			fp0 = lut2bit[buf[i]];
			i++;
			fp1 = lut2bit[buf[i]];
			i++;
			fp2 = lut2bit[buf[i]];
			i++;
			fp3 = lut2bit[buf[i]];
			i++;
		}

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp1[0];
		data[5][o]  = fp1[1];
		data[6][o]  = fp1[2];
		data[7][o]  = fp1[3];
		data[8][o]  = fp2[0];
		data[9][o]  = fp2[1];
		data[10][o] = fp2[2];
		data[11][o] = fp2[3];
		data[12][o] = fp3[0];
		data[13][o] = fp3[1];
		data[14][o] = fp3[2];
		data[15][o] = fp3[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_32bitstream_2bit_decimation2(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1, *fp2, *fp3;
	int o, i;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = fp1 = fp2 = fp3 = zeros;
			nblank++;
			i += 8;
		}
		else
		{
			fp0 = lut2bit[buf[i]];
			i++;
			fp1 = lut2bit[buf[i]];
			i++;
			fp2 = lut2bit[buf[i]];
			i++;
			fp3 = lut2bit[buf[i]];
			i += 5;
		}

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp1[0];
		data[5][o]  = fp1[1];
		data[6][o]  = fp1[2];
		data[7][o]  = fp1[3];
		data[8][o]  = fp2[0];
		data[9][o]  = fp2[1];
		data[10][o] = fp2[2];
		data[11][o] = fp2[3];
		data[12][o] = fp3[0];
		data[13][o] = fp3[1];
		data[14][o] = fp3[2];
		data[15][o] = fp3[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

static int mark5b_decode_32bitstream_2bit_decimation4(struct mark5_stream *ms,
	int nsamp, float **data)
{
	uint8_t *buf;
	float *fp0, *fp1, *fp2, *fp3;
	int o, i, df;
	int nblank = 0;

	buf = ms->payload;
	i = ms->readposition;
	df = ms->decimation*4 - 3;

	for(o = 0; o < nsamp; o++)
	{
		if(i <  ms->blankzonestartvalid[0] ||
		   i >= ms->blankzoneendvalid[0])
		{
			fp0 = zeros;
			i++;
			fp1 = zeros;
			i++;
			fp2 = zeros;
			i++;
			fp3 = zeros;
			nblank++;
		}
		else
		{
			fp0 = lut2bit[buf[i]];
			i++;
			fp1 = lut2bit[buf[i]];
			i++;
			fp2 = lut2bit[buf[i]];
			i++;
			fp3 = lut2bit[buf[i]];
		}
		i += df;

		data[0][o]  = fp0[0];
		data[1][o]  = fp0[1];
		data[2][o]  = fp0[2];
		data[3][o]  = fp0[3];
		data[4][o]  = fp1[0];
		data[5][o]  = fp1[1];
		data[6][o]  = fp1[2];
		data[7][o]  = fp1[3];
		data[8][o]  = fp2[0];
		data[9][o]  = fp2[1];
		data[10][o] = fp2[2];
		data[11][o] = fp2[3];
		data[12][o] = fp3[0];
		data[13][o] = fp3[1];
		data[14][o] = fp3[2];
		data[15][o] = fp3[3];

		if(i >= MK5B_PAYLOADSIZE)
		{
			if(mark5_stream_next_frame(ms) < 0)
			{
				return -1;
			}
			buf = ms->payload;
			i = 0;
		}
	}

	ms->readposition = i;

	return nsamp - nblank;
}

/******************************************************************/

static int mark5_format_mark5b_make_formatname(struct mark5_stream *ms)
{
	if(ms->format == MK5_FORMAT_MARK5CB)
	{
		sprintf(ms->formatname, "Mark5CB-%d-%d-%d", ms->Mbps, 
			ms->nchan, ms->nbit);
	}
	else
	{
		sprintf(ms->formatname, "Mark5B-%d-%d-%d", ms->Mbps, 
			ms->nchan, ms->nbit);
	}

	return 0;
}

static int mark5_format_mark5b_init(struct mark5_stream *ms)
{
	struct mark5_format_mark5b *f;
	int mjd1, sec1, ns1;
	double dns, dns1;
	int datarate;
	int bytes;
	int k, df, framenum;

	if(!ms)
	{
		fprintf(stderr, "mark5_format_mark5b_init: ms = 0\n");
		return -1;
	}

	f = (struct mark5_format_mark5b *)(ms->formatdata);

	ms->samplegranularity = 8/(f->nbitstream*ms->decimation);
	if(ms->samplegranularity <= 0)
	{
		ms->samplegranularity = 1;
	}
	ms->framebytes = 10016;
	ms->databytes = 10000;
	ms->payloadoffset = 16;
	ms->framesamples = ms->databytes*8/(f->nbitstream*ms->decimation);
	ms->blanker = blanker_mark5;
	if(ms->Mbps > 0)
	{
		ms->framens = 80000000.0/ms->Mbps;
	}

	mark5_format_mark5b_make_formatname(ms);

	if(ms->datawindow)
	{
		if(ms->datawindowsize < ms->framebytes)
		{
			return -1;
		}

		/* look through entire data window, up to 1Mibytes */
		bytes = ms->datawindowsize < (1<<20) ?
			ms->datawindowsize : (1<<20);

		/* first look for normal Mark5B sync word */
		ms->frameoffset = findfirstframe(ms->datawindow, bytes,
			mark5bSync);

		if(ms->frameoffset >= 0)
		{
			ms->format = MK5_FORMAT_MARK5B;
		}
		else if(ms->Mbps > 0)
		{
			/* then look for Mark5C compatibility mode sync word */
			ms->frameoffset = findfirstframe(ms->datawindow, bytes,
				mark5cCompatSync);
			if(ms->frameoffset >= 0)
			{
				ms->format = MK5_FORMAT_MARK5CB;
				ms->gettime = mark5_stream_frame_time_mark5cb;
				ms->framens = 80000000/ms->Mbps;
			}
			else
			{
				return -1;
			}
		}
		else 
		{
			return -1;
		}

		ms->frame = ms->datawindow + ms->frameoffset;
		ms->payload = ms->frame + ms->payloadoffset;

		ms->gettime(ms, &ms->mjd, &ms->sec, &dns);
		ms->ns = (int)(dns + 0.5);

		if(ms->Mbps > 0)
		{
			ms->samprate = ms->framesamples*
				(1000000000.0/ms->framens);
		}
		else
		{
			k = 8;
			while((k+2)*ms->framebytes > ms->datawindowsize && k>1)
			{
				k /= 2;
			}
			ms->frame += k*ms->framebytes;
			ms->gettime(ms, &mjd1, &sec1, &dns1);
			ns1 = (int)(dns1 + 0.5);
			ms->frame -= k*ms->framebytes;

			/* assume frame time less than 1 second, integer number
			 * of frames per second
			 */
			if(ns1 != ms->ns)
			{
				ms->framens = (ns1 - ms->ns)/k;

				/* get time again so ms->framens is used */
				ms->gettime(ms, &ms->mjd, &ms->sec, &dns);
				ms->ns = (int)(dns + 0.5);

				if(ms->framens <= 0)
				{
					ms->framens += 1000000000;
				}
				ms->samprate = ms->framesamples*
					(1000000000/ms->framens);
				datarate = ms->samprate*ms->nbit*ms->nchan/1000000;
				if(datarate != ms->Mbps)
				{
					if(ms->Mbps > 0)
					{
						fprintf(stderr, "Warning -- data rate "
							"disagrees : %d != %d\n",
							datarate, ms->Mbps);
					}
					ms->Mbps = datarate;
				}
			}
			else
			{
				fprintf(stderr, "Warning -- rate calc. "
					"suspect ns0=ns1=%d k=%d\n", ns1, k);
			}
		}
	}

	/* see if we need to advance a small number of frames to make the
	 * first one start at integer nanosec
	 */
	k = ms->Mbps/1024;
	if(k > 0)
	{
		if(ms->datawindow)
		{
			framenum = mark5_stream_frame_num_mark5b(ms);
			df = k - framenum % k;
			if(df != k)
			{
				ms->frame += df*ms->framebytes;
				ms->frameoffset += df*ms->framebytes;
				ms->gettime(ms, &ms->mjd, &ms->sec, &dns);
				ms->ns = (int)(dns + 0.5);
			}
		}
		ms->framegranularity = k;
	}
	else
	{
		ms->framegranularity = 1;
	}

	ms->gframens = (int)(ms->framegranularity*ms->framens + 0.5);

	return 0;
}

static int mark5_format_mark5b_final(struct mark5_stream *ms)
{
	if(!ms)
	{
		return -1;
	}

	if(ms->formatdata)
	{
		free(ms->formatdata);
	}

	return 0;
}

static int one(const struct mark5_stream *ms)
{
	return 1;
}

struct mark5_format_generic *new_mark5_format_mark5b(int Mbps,
	int nchan, int nbit, int decimation)
{
	static int first = 1;
	struct mark5_format_generic *f;
	struct mark5_format_mark5b *m;
	int decoderindex = 0;
	int nbitstream;

	nbitstream = nchan*nbit;

	if(first)
	{
		initluts();
		first = 0;
	}

	if(decimation == 1)
	{
		decoderindex += 0;
	}
	else if(decimation == 2)
	{
		decoderindex += 12;
	}
	else if(decimation % 4 == 0)  /* all mults of 4 */
	{
		decoderindex += 24;
	}
	else
	{
		fprintf(stderr, "decimation must be 1, 2 or a mult of 4\n");
	}

	if(nbit == 1)
	{
		decoderindex += 0;
	}
	else if(nbit == 2)
	{
		decoderindex += 6;
	}
	else
	{
		fprintf(stderr, "new_mark5_format_mark5b : "
			"nbit needs to be 1 or 2\n");
		return 0;
	}
	
	if(nbitstream == 1)
	{
		decoderindex += 0;
	}
	else if(nbitstream == 2)
	{
		decoderindex += 1;
	}
	else if(nbitstream == 4)
	{
		decoderindex += 2;
	}
	else if(nbitstream == 8)
	{
		decoderindex += 3;
	}
	else if(nbitstream == 16)
	{
		decoderindex += 4;
	}
	else if(nbitstream == 32)
	{
		decoderindex += 5;
	}
	else
	{
		fprintf(stderr, "new_mark5_format_mark5b : "
			"nbitstream needs to be 1, 2, 4, 8, 16 or 32\n");
		return 0;
	}

	if(decoderindex == 6)
	{
		fprintf(stderr, "Illegal format\n");
		return 0;
	}

	m = (struct mark5_format_mark5b *)calloc(1, 
		sizeof(struct mark5_format_mark5b));
	f = (struct mark5_format_generic *)calloc(1, 
		sizeof(struct mark5_format_generic));

	m->nbitstream = nbitstream;

	f->Mbps = Mbps;
	f->nchan = nchan;
	f->nbit = nbit;
	f->formatdata = m;
	f->gettime = mark5_stream_frame_time_mark5b;
	f->init_format = mark5_format_mark5b_init;
	f->final_format = mark5_format_mark5b_final;
	f->fixmjd = mark5_format_mark5b_fixmjd;
	f->validate = one;
	f->decimation = decimation;
	f->decode = 0;
	switch(decoderindex)
	{
		case 0 : f->decode = mark5b_decode_1bitstream_1bit_decimation1; break;
		case 1 : f->decode = mark5b_decode_2bitstream_1bit_decimation1; break;
		case 2 : f->decode = mark5b_decode_4bitstream_1bit_decimation1; break;
		case 3 : f->decode = mark5b_decode_8bitstream_1bit_decimation1; break;
		case 4 : f->decode = mark5b_decode_16bitstream_1bit_decimation1; break;
		case 5 : f->decode = mark5b_decode_32bitstream_1bit_decimation1; break;

		case 7 : f->decode = mark5b_decode_2bitstream_2bit_decimation1; break;
		case 8 : f->decode = mark5b_decode_4bitstream_2bit_decimation1; break;
		case 9 : f->decode = mark5b_decode_8bitstream_2bit_decimation1; break;
		case 10: f->decode = mark5b_decode_16bitstream_2bit_decimation1; break;
		case 11: f->decode = mark5b_decode_32bitstream_2bit_decimation1; break;
		case 12: f->decode = mark5b_decode_1bitstream_1bit_decimation2; break;
		case 13: f->decode = mark5b_decode_2bitstream_1bit_decimation2; break;
		case 14: f->decode = mark5b_decode_4bitstream_1bit_decimation2; break;
		case 15: f->decode = mark5b_decode_8bitstream_1bit_decimation2; break;
		case 16: f->decode = mark5b_decode_16bitstream_1bit_decimation2; break;
		case 17: f->decode = mark5b_decode_32bitstream_1bit_decimation2; break;

		case 19: f->decode = mark5b_decode_2bitstream_2bit_decimation2; break;
		case 20: f->decode = mark5b_decode_4bitstream_2bit_decimation2; break;
		case 21: f->decode = mark5b_decode_8bitstream_2bit_decimation2; break;
		case 22: f->decode = mark5b_decode_16bitstream_2bit_decimation2; break;
		case 23: f->decode = mark5b_decode_32bitstream_2bit_decimation2; break;
		case 24: /* special case needing explicit decimation4 case */
			if(decimation == 4)
			 f->decode = mark5b_decode_1bitstream_1bit_decimation4; 
			else if(decimation % 8 == 0)
			 f->decode = mark5b_decode_1bitstream_1bit_decimation8; 
			break;
		case 25: f->decode = mark5b_decode_2bitstream_1bit_decimation4; break;
		case 26: f->decode = mark5b_decode_4bitstream_1bit_decimation4; break;
		case 27: f->decode = mark5b_decode_8bitstream_1bit_decimation4; break;
		case 28: f->decode = mark5b_decode_16bitstream_1bit_decimation4; break;
		case 29: f->decode = mark5b_decode_32bitstream_1bit_decimation4; break;

		case 31: f->decode = mark5b_decode_2bitstream_2bit_decimation4; break;
		case 32: f->decode = mark5b_decode_4bitstream_2bit_decimation4; break;
		case 33: f->decode = mark5b_decode_8bitstream_2bit_decimation4; break;
		case 34: f->decode = mark5b_decode_16bitstream_2bit_decimation4; break;
		case 35: f->decode = mark5b_decode_32bitstream_2bit_decimation4; break;
	}

	if(f->decode == 0)
	{
		fprintf(stderr, "Illegal combination of decimation, bitstreams and bits\n");
		return 0;
	}

	return f;
}
