#include <stdio.h>
#include <math.h>
#include <glib.h>
#include <wfblib/vecarray.h>
#include <wfblib/polygon.h>
#include <wfblib/randdist.h>

/* returns a list of points contained within polygon p */
VecArray calcintpoints(const VecArray p, int nintpoints)
{
	VecArray q;
	int i, c;
	double a, b;
	double x1, x2, y1, y2;

	/* bounds on p -- q points are inside p points */
	Vectorminmax(p[0], &x1, &x2);
	Vectorminmax(p[1], &y1, &y2);
	
	q = newpopulatedVecArray(2, nintpoints);
	fprintf(stderr, "%d internal points:\n", nintpoints);
	for(i = 0; i < nintpoints; )
	{
		a = x1 + (x2-x1)*rand_pm_one();
		b = y1 + (y2-y1)*rand_pm_one();
		c = polygonside(p[0], p[1], a, b);
		if(c == 1)
		{
			q[0][i] = a;
			q[1][i] = b;
			i++;
			fprintf(stderr, "%f %f\n", a, b);
		}
	}
	fprintf(stderr, "\n");

	return q;
}

void computenormal(VecArray p)
{
	int i, n;
	double dx, dy, mag, l=0.0;
	
	n = VecArrayVectorSize(p);
	
	for(i = 0; i < n; i++)
	{
		p[8][i] = l;
		dx = p[2][i] - p[0][i];
		dy = p[3][i] - p[1][i];
		mag = sqrt(dx*dx + dy*dy);
		l += mag;
		dx /= mag;
		dy /= mag;
		p[6][i] = dy;
		p[7][i] = -dx;
	}
	scaleVector(p[8], 2.0/l);
}

/* return x1, y1, x2, y2, x0, y0, nx, ny, S */

VecArray testellipse(int n, double a, double b)
{
	int i;
	double theta;
	VecArray p;

	p = newpopulatedVecArray(9, n);
	
	for(i = 0; i < n; i++)
	{
		theta = 2.0*M_PI*i/n;
		p[0][i] = a*cos(theta);
		p[1][i] = b*sin(theta);
	}

	p[2][n-1] = p[0][0];
	p[3][n-1] = p[1][0];
	
	for(i = 1; i < n; i++)
	{
		p[2][i-1] = p[0][i];
		p[3][i-1] = p[1][i];
	}

	for(i = 0; i < n; i++)
	{
		p[4][i] = 0.5*(p[0][i] + p[2][i]);
		p[5][i] = 0.5*(p[1][i] + p[3][i]);
	}

	computenormal(p);

	VecArraytofile(p, "ellipse.out", 0);
	
	return p;
}

int addsegment(VecArray p, int p0,
	double x0, double y0, double x1, double y1, double d)
{
	int i, n;
	double dx, dy;
	
	n = sqrt((x1-x0)*(x1-x0) + (y1-y0)*(y1-y0))/d + 1.0;
	
	g_assert(VecArrayVectorSize(p) > p0 + n);

	dx = (x1-x0)/n;
	dy = (y1-y0)/n;

	for(i = 0; i < n; i++)
	{
		p[0][p0] = x0 + dx * i;
		p[1][p0] = y0 + dy * i;
		p0++;
	}

	return p0;	
}

VecArray testpolygon(const Vector X, const Vector Y, double dx)
{
	VecArray p;
	int i, n, m=0;

	n = VectorSize(X);
	g_assert(VectorSize(Y) == n);
	
	p = newpopulatedVecArray(9, 10000);
	
	for(i = 1; i < n; i++)
		m = addsegment(p, m, X[i-1], Y[i-1], X[i], Y[i], dx);
	m = addsegment(p, m, X[n-1], Y[n-1], X[0], Y[0], dx);

	VecArrayresize(p, m);
	
	p[2][m-1] = p[0][0];
	p[3][m-1] = p[1][0];
	
	for(i = 1; i < m; i++)
	{
		p[2][i-1] = p[0][i];
		p[3][i-1] = p[1][i];
	}

	for(i = 0; i < m; i++)
	{
		p[4][i] = 0.5*(p[0][i] + p[2][i]);
		p[5][i] = 0.5*(p[1][i] + p[3][i]);
	}

	computenormal(p);
	
	VecArraytofile(p, "polygon.out", 0);
	
	return p;
}
	
static void closepoly(VecArray P, int first, int last)
{
	int i;

	P[2][last] = P[0][first];
	P[3][last] = P[1][first];
	
	for(i = first; i < last; i++)
	{
		P[2][i] = P[0][i+1];
		P[3][i] = P[1][i+1];
	}

	for(i = first; i <= last; i++)
	{
		P[4][i] = 0.5*(P[0][i] + P[2][i]);
		P[5][i] = 0.5*(P[1][i] + P[3][i]);
	}
}

VecArray polygonfromfile(const char *filename, double dx)
{
	VecArray P, G;
	FILE *in;
	int p0=0, p=0, n=0, N=0;
	char line[256];
	double x0, y0, x1, y1;

	in = fopen(filename, "r");
	if(!in)
	{
		fprintf(stderr, "Error opening %s for read\n", filename);
		return 0;
	}

	P = newpopulatedVecArray(9, 10000);

	for(;;)
	{
		fgets(line, 255, in);
		if(feof(in)) break;
		if(line[0] == '#') continue;
		if(sscanf(line, "%lf%lf", &x1, &y1) != 2)
		{
			if(n == 0) continue;
			if(n < 3)
			{
				fprintf(stderr, "%s is malformed\n", filename);
				deleteVecArrayandVectors(P);
				fclose(in);
				return 0;
			}
			p0 = addsegment(P, p0, x0, y0, P[0][p], P[1][p], dx);
			closepoly(P, p, p0-1);
			n = 0;
			p = p0;
		}
		else
		{
			N++;
			if(n) p0 = addsegment(P, p0, x0, y0, x1, y1, dx);
			n++;
			x0 = x1;
			y0 = y1;
		}
		
	}
	fclose(in);
	if(n < 3)
	{
		fprintf(stderr, "%s is malformed\n", filename);
		deleteVecArrayandVectors(P);
		return 0;
	}
	p0 = addsegment(P, p0, x0, y0, P[0][p], P[1][p], dx);
	closepoly(P, p, p0-1);
	
	VecArrayresize(P, p0);

	computenormal(P);
	
	G = newVecArray(2);
	G[0] = P[0];
	G[1] = P[1];
	
	fprintf(stderr, "Polygon loaded from %s:\n", filename);
	fprintf(stderr, "  Number of vertices   : %d\n", N);
	fprintf(stderr, "  Number of segments   : %d\n", p0);

	sprintf(line, "%s.polygon", filename);
	
	VecArraytofile(G, line, "%f");
	deleteVecArray(G);

	return P;
}

VecArray testrect(int nx, int ny, double d)
{
	VecArray p;
	int i, n=0;
	double mx, my;

	mx = 0.5*nx*d;
	my = 0.5*ny*d;
	
	p = newpopulatedVecArray(9, nx+nx+ny+ny);
	for(i = 0; i < nx; i++)
	{
		p[0][n] = -mx + d*i;
		p[1][n] = -my;
		n++;
	}
	for(i = 0; i < ny; i++)
	{
		p[0][n] = mx;
		p[1][n] = -my + d*i;
		n++;
	}
	for(i = 0; i < nx; i++)
	{
		p[0][n] = mx - d*i;
		p[1][n] = my;
		n++;
	}
	for(i = 0; i < ny; i++)
	{
		p[0][n] = -mx;
		p[1][n] = my - d*i;
		n++;
	}

	g_assert(n == nx+nx+ny+ny);

	p[2][n-1] = p[0][0];
	p[3][n-1] = p[1][0];
	
	for(i = 1; i < n; i++)
	{
		p[2][i-1] = p[0][i];
		p[3][i-1] = p[1][i];
	}

	for(i = 0; i < n; i++)
	{
		p[4][i] = 0.5*(p[0][i] + p[2][i]);
		p[5][i] = 0.5*(p[1][i] + p[3][i]);
	}

	computenormal(p);

	VecArraytofile(p, "square.out", 0);
	
	return p;
}
