#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <wfblib/mathfunc.h>
#include <wfblib/vector.h>
#include <wfblib/vecarray.h>
#include <wfblib/zvector.h>
#include <wfblib/lapack.h>
#include <wfblib/polygon.h>
#include <wfblib/randdist.h>
#include "shapes.h"

const double eta=120.0*M_PI;	/* impedence of vaccuum */

/* returns width in the y direction of polygon p */
double width(const VecArray p)
{
	VectorType w0, w1;

	Vectorminmax(p[1], &w0, &w1);

	return w1-w0;
}

ZMatrix computeTM(const VecArray p, int ndiv, double lambda)
{
	int i, j, m, nd2, d;
	double x0, y0, dx, dy, dl, k, x, y, arg, re, im;
	ZMatrix M;
	
	k = 2.0*M_PI/lambda;
	
	if(ndiv % 2 == 0) ndiv++;
	nd2 = (ndiv-1)/2;

	m = VecArrayVectorSize(p);
	
	M = newZMatrix(m, m);
	
	for(i = 0; i < m; i++)	/* destination == field */
	{
		x0 = p[4][i];
		y0 = p[5][i];

		for(j = 0; j < m; j++) /* source == current */
		{
			re = im = 0.0;
			
			dx = (p[2][j]-p[0][j])/ndiv;
			dy = (p[3][j]-p[1][j])/ndiv;
			dl = sqrt(dx*dx + dy*dy);

			for(d = 0; d < ndiv; d++) /* current sub-segment */
			{
				if(d == nd2 && i == j)
				{
					re += 1.0;
					im -= (2.0/M_PI)*
						(log(k*dl/4.0)+C_EULER-1.0);
				}
				else
				{
					x = p[0][j] + ((double)d+0.5)*dx - x0;
					y = p[1][j] + ((double)d+0.5)*dy - y0;
					arg = k*sqrt(x*x + y*y);
					re += bessel_J0(arg);
					im -= bessel_Y0(arg);
				}
			}

			re *= dl;
			im *= dl;
			
			M[i][j].re = re;
			M[i][j].im = im;
		}
	}

	/* Scale array by k eta / 4 */
	scaleZMatrix(M, k*eta/4.0, 0.0);

	return M;
}

/* compute complex matrix M that relates a current at one vertex of the
 * input polygon to the electric field at another.
 */
ZMatrix computeTE(const VecArray p, int ndiv, double lambda)
{
	int i, j, m, nd2, d;
	double x0, y0, dx, dy, dl, k, x, y, re, im, mag, ndotR;
	ZMatrix M;
	
	k = 2.0*M_PI/lambda;
	
	if(ndiv % 2 == 0) ndiv++;
	nd2 = (ndiv-1)/2;

	m = VecArrayVectorSize(p);
	
	M = newZMatrix(m, m);
	
	for(i = 0; i < m; i++)	/* destination == field */
	{
		x0 = p[4][i];
		y0 = p[5][i];

		for(j = 0; j < m; j++)	/* source == current */
		{
			re = im = 0.0;
		
			dx = (p[2][j] - p[0][j])/ndiv;
			dy = (p[3][j] - p[1][j])/ndiv;
			dl = sqrt(dx*dx + dy*dy);

			for(d = 0; d < ndiv; d++) /* current sub-segment */
			{
				if(d == nd2 && i == j)
				{
					re += 0.5;
				}
				else
				{
					x = x0 - (p[0][j] + ((double)d+0.5)*dx);
					y = y0 - (p[1][j] + ((double)d+0.5)*dy);
					mag = sqrt(x*x + y*y);

					ndotR = (p[6][j]*x + p[7][j]*y)/mag;

					re += 0.25*dl*ndotR*k*bessel_Y1(k*mag);
					im += 0.25*dl*ndotR*k*bessel_J1(k*mag);
				}
			}

			M[i][j].re = re;
			M[i][j].im = im;
		}
	}

	scaleZMatrix(M, eta, 0);

	return M;
}

/* find currents using previously computed matrix M for polygon p given a
 * unit strength plane wave coming from the negative x direction.
 */
ZVector currents(const ZMatrix M, const VecArray p, double lambda)
{
	ZVector u, v;
	int i, n, m;
	double phase;
	ZMatrix w;

	n = ZMatrixSize1(M);
	m = ZMatrixSize2(M);

	g_assert(m == n);
	g_assert(m == VecArrayVectorSize(p));
	
	u = newZVector(n);

	/* incident wave with unit amplitude */
	for(i = 0; i < n; i++)
	{
		phase = 2.0*M_PI*p[4][i]/lambda;
		u[i].re =  cos(phase);  /* Re[E] */
		u[i].im = -sin(phase);  /* Im[E] */
	}

	w = ZMatrixLUinverse(M);
	v = ZMatrixZVectormultiply(w, u);
	deleteZMatrix(w);
	deleteZVector(u);

	return v;
}

/* calculates the far field complex electric field at n equally spaced
 * points on the circle at infinity multiplied by kr for a polygon p with
 * currents at each vertex given by I at wavelength lambda.
 */
ZVector scatterTM(const VecArray p, const ZVector I, int n, double lambda)
{
	double theta;
	double dtheta;
	double sintheta, costheta, ckl, skl;
	double k, x0, y0, dx, dy, dl, l, Er, Ei, f;
	ZVector field;
	int i, j, m;

	k = 2.0*M_PI/lambda;
	f = k*eta/sqrt(8.0*M_PI);
	dtheta = 2.0*M_PI/n;
	m = VecArrayVectorSize(p);
	field = newZVector(n);

	for(i = 0; i < n; i++)
	{
		theta = i*dtheta;
		sintheta = sin(theta);
		costheta = cos(theta);
		
		Er = 0.0;
		Ei = 0.0;
		
		for(j = 0; j < m; j++)
		{
			dx = (p[2][j] - p[0][j]);
			dy = (p[3][j] - p[1][j]);
			dl = sqrt(dx*dx + dy*dy);
			x0 = p[4][j];
			y0 = p[5][j];
			l = x0*costheta + y0*sintheta;
			ckl = cos(k*l);
			skl = sin(k*l);
			Er += dl*(I[j].re*ckl - I[j].im*skl);
			Ei += dl*(I[j].re*skl + I[j].im*ckl);
		}
		
		field[i].re = f*Er;
		field[i].im = f*Ei;
	}

	return field;
}

/* calculates the far field complex electric field at n equally spaced
 * points on the circle at infinity multiplied by kr for a polygon p with
 * currents at each vertex given by I at wavelength lambda.
 */
ZVector scatterTE(const VecArray p, const ZVector I, int n, double lambda)
{
	double theta;
	double dtheta;
	double sintheta, costheta, ckl, skl;
	double k, x0, y0, dx, dy, dl, l, Er, Ei, ndotR, f;
	ZVector field;
	int i, j, m;

	k = 2.0*M_PI/lambda;
	f = k*eta/sqrt(8.0*M_PI);
	dtheta = 2.0*M_PI/n;
	m = VecArrayVectorSize(p);
	field = newZVector(n);

	for(i = 0; i < n; i++)
	{
		theta = i*dtheta;
		sintheta = sin(theta);
		costheta = cos(theta);
		
		Er = 0.0;
		Ei = 0.0;

		for(j = 0; j < m; j++)
		{
			dx = p[2][j] - p[0][j];
			dy = p[3][j] - p[1][j];
			dl = sqrt(dx*dx + dy*dy);
			x0 = p[4][j];
			y0 = p[5][j];
					
			ndotR = p[6][j]*costheta + p[7][j]*sintheta;

			l = x0*costheta + y0*sintheta;
			ckl = cos(k*l);
			skl = sin(k*l);
			Er += dl*ndotR*(I[j].re*ckl - I[j].im*skl);
			Ei += dl*ndotR*(I[j].re*skl + I[j].im*ckl);
		}
		
		field[i].re = f*Er;
		field[i].im = f*Ei;
	}

	return field;
}

/* computes IFR_H given a polygon p, wavelength lambda 
 *
 * nseg is the number of segments to break the current segment into when
 * integrating.  Should probably be between 5 and 100.
 */
double IFR_E(const VecArray p, double lambda, int nseg)
{
	ZMatrix M;
	ZVector I;
	ZVector field;
	double ifr, k;

	k = 2.0*M_PI/lambda;

	M = computeTM(p, nseg, lambda);
	I = currents(M, p, lambda);
	field = scatterTM(p, I, 360, lambda);

	ifr = sqrt(field[0].re*field[0].re + field[0].im*field[0].im) /
		(k*2.0*width(p));

	deleteZMatrix(M);
	deleteZVector(I);
	deleteZVector(field);

	return ifr*M_PI*M_PI/2.0;
}

/* computes IFR_H given a polygon p, wavelength lambda 
 *
 * nseg is the number of segments to break the current segment into when
 * integrating.  Should probably be between 5 and 100.
 */
double IFR_H(const VecArray p, double lambda, int nseg)
{
	ZMatrix M;
	ZVector I;
	ZVector field;
	double ifr, k;

	k = 2.0*M_PI/lambda;

	M = computeTE(p, nseg, lambda);
	I = currents(M, p, lambda);
	field = scatterTE(p, I, 360, lambda);

	ifr = sqrt(field[0].re*field[0].re + field[0].im*field[0].im) /
		(k*2.0*width(p));

	deleteZMatrix(M);
	deleteZVector(I);
	deleteZVector(field);

	return ifr*M_PI*M_PI/2.0;
}

void save_fc(FILE *out, double r, const ZVector field)
{
	int i, n;

	n = ZVectorSize(field);
	fprintf(out, "# r = %f\n", r);
	for(i = 0; i < n; i++) fprintf(out, "%e %e\n", 
		field[i].re, field[i].im);
	fprintf(out, "\n");
}

/* computes complex IFR E and H for a range of width/lambda, saving the
 * ifr values currents and fields to files
 */
void ifrcompute(const VecArray p, double r1, double r2, double dr, 
	const char *prefix, int savefields, int savecurrents)
{
	ZMatrix M;
	ZVector I;
	ZVector field;
	double w, r, k, lambda;
	double ifr_e_re, ifr_e_im, ifr_h_re, ifr_h_im;
	double ifr_e, ifr_h;
	FILE *ifr, *tmf, *tef, *tmc, *tec;
	char fn[256];
	const int nseg=31;

	sprintf(fn, "%s.ifr", prefix);
	ifr = fopen(fn, "w");
	if(!ifr)
	{
		fprintf(stderr, "cannot open %s for write\n", fn);
		return;
	}

	if(savefields)
	{
		sprintf(fn, "%s.TM.field", prefix);
		tmf = fopen(fn, "w");
		if(!tmf)
		{
			fprintf(stderr, "cannot open %s for write\n", fn);
			return;
		}
		sprintf(fn, "%s.TE.field", prefix);
		tef = fopen(fn, "w");
		if(!tef)
		{
			fprintf(stderr, "cannot open %s for write\n", fn);
			return;
		}
	}
	
	if(savecurrents)
	{
		sprintf(fn, "%s.TM.currents", prefix);
		tmc = fopen(fn, "w");
		if(!tmc)
		{
			fprintf(stderr, "cannot open %s for write\n", fn);
			return;
		}
		sprintf(fn, "%s.TE.currents", prefix);
		tec = fopen(fn, "w");
		if(!tec)
		{
			fprintf(stderr, "cannot open %s for write\n", fn);
			return;
		}
	}

	w = width(p);

	fprintf(stderr, "\nw/lambda\tIFR_E\t\tIFR_H\n");
	fflush(stderr);

	for(r = r1; r <= r2; r+=dr)
	{
		lambda = w/r;
		k = 2.0*M_PI/lambda;

		M = computeTM(p, nseg, lambda);
		I = currents(M, p, lambda);
		field = scatterTM(p, I, 360, lambda);
		ifr_e_re = field[0].re*M_PI*M_PI/(4.0*w*k);
		ifr_e_im = field[0].im*M_PI*M_PI/(4.0*w*k);
		if(savefields) save_fc(tmf, r, field);
		if(savecurrents) save_fc(tmc, r, I);
		deleteZMatrix(M);
		deleteZVector(I);
		deleteZVector(field);
		
		M = computeTE(p, nseg, lambda);
		I = currents(M, p, lambda);
		field = scatterTE(p, I, 360, lambda);
		ifr_h_re = field[0].re*M_PI*M_PI/(4.0*w*k);
		ifr_h_im = field[0].im*M_PI*M_PI/(4.0*w*k);
		if(savefields) save_fc(tef, r, field);
		if(savecurrents) save_fc(tec, r, I);
		deleteZMatrix(M);
		deleteZVector(I);
		deleteZVector(field);
		
		ifr_e = sqrt(ifr_e_re*ifr_e_re + ifr_e_im*ifr_e_im);
		ifr_h = sqrt(ifr_h_re*ifr_h_re + ifr_h_im*ifr_h_im);
		
		fprintf(ifr, "%f %f %f %f %f %f %f\n", r, ifr_e, ifr_h,
			ifr_e_re, ifr_e_im, ifr_h_re, ifr_h_im);
		
		printf("%f\t%f\t%f\n", r, ifr_e, ifr_h);
	}

	fclose(ifr);
	if(savefields)
	{
		fclose(tmf);
		fclose(tef);
	}
	if(savecurrents)
	{
		fclose(tmc);
		fclose(tec);
	}
}

int usage(const char *program)
{
	fprintf(stderr, "\n");
	fprintf(stderr, "ifr2d : a program for strut IFR calculation in 2d\n");
	fprintf(stderr, "\n  W. Brisken, Nov 24, 2003\n\n");
	fprintf(stderr, "usage : %s <inputfile> { -r <r> | [-r1 <r1>] "
			"[-r2 <r2>] [-dr <dr>] [-dx <dx>] }\n\n", program);
	fprintf(stderr, "  Computes ifr values for struts.\n");
	fprintf(stderr, "  r, the ratio of width to wavelength, "
			"ranges from\n");
	fprintf(stderr, "  <r1> to <r2> in steps of <dr>, or is set at <r>\n");
	fprintf(stderr, "  Gridding of the strut is done with segments no "
			"larger than <dx>.\n\n");
	fprintf(stderr, "  Default values:\n");
	fprintf(stderr, "    <r1> = 0.05\n");
	fprintf(stderr, "    <r2> = 2.00\n");
	fprintf(stderr, "    <dr> = 0.05\n");
	fprintf(stderr, "    <dx> = 0.1\n");
	return 0;
}

int main(int argc, char **argv)
{
	VecArray p;
	double r1, r2, dr, dx, area;
	char *prefix;
	int i, n;

	if(argc < 2) return usage(argv[0]);

	prefix = argv[1];

	r1 = 0.05;
	r2 = 2.001;
	dr = 0.05;
	dx = 0.1;

	if(argc > 2) for(i = 2; i < argc; i++)
	{
		if(strcmp(argv[i], "-r1") == 0 && argc > i+1)
		{
			r1 = atof(argv[++i]);
			if(r1 <= 0) return usage(argv[0]);
		}
		if(strcmp(argv[i], "-r") == 0 && argc > i+1)
		{
			r1 = r2 = atof(argv[++i]);
			if(r1 <= 0) return usage(argv[0]);
		}
		else if(strcmp(argv[i], "-r2") == 0 && argc > i+1)
		{
			r2 = atof(argv[++i]);
			if(r2 <= 0) return usage(argv[0]);
		}
		else if(strcmp(argv[i], "-dr") == 0 && argc > i+1)
		{
			dr = atof(argv[++i]);
			if(dr <= 0) return usage(argv[0]);
		}
		else if(strcmp(argv[i], "-dx") == 0 && argc > i+1)
		{
			dx = atof(argv[++i]);
			if(dx <= 0) return usage(argv[0]);
		}
		else return usage(argv[0]);
	}

	if(r1 > r2) return usage(argv[0]);

	p = polygonfromfile(argv[1], dx);

	if(!p) return 0;

	n = VecArrayVectorSize(p);
	if(n < 10)
	{
		fprintf(stderr,"*** WARNING : gridding is perhaps to coarse\n");
		fprintf(stderr, "    Calculation may not be accurate\n");
		fprintf(stderr, "    Consider using a smaller value of dx\n");
	}
	if(n > 1000)
	{
		fprintf(stderr, "*** WARNING : gridding is perhaps to fine\n");
		fprintf(stderr, "    Calculation will take a long time.\n");
		fprintf(stderr, "    Consider using a larger value of dx\n");
	}

	area = polygonarea(p[0], p[1]);
	
	if(area < 0.0)
	{
		fprintf(stderr, "*** ERROR : polygon must be defined in a "
				"clockwise manner\n");
		deleteVecArrayandVectors(p);
		return 0;
	}
	
	fprintf(stderr, "  Cross sectional area : %f\n", area);
	fprintf(stderr, "  Perimeter            : %f\n", 
		polygonperimeter(p[0], p[1]));
	fprintf(stderr, "r1 = %f r2 = %f dr = %f dx =%f\n", r1, r2, dr, dx);

	ifrcompute(p, r1, r2, dr, prefix, 1, 1);

	deleteVecArrayandVectors(p);

	return 0;
}
